"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.PM = void 0;
exports.getPackageManagerCommand = getPackageManagerCommand;
exports.detectPackageManagerByEnv = detectPackageManagerByEnv;
exports.detectPackageManagerByLockfile = detectPackageManagerByLockfile;
exports.detectYarnBerry = detectYarnBerry;
const path = require("path");
const fs = require("fs");
const which = require("which");
const child_process_1 = require("child_process");
var PM;
(function (PM) {
    PM["NPM"] = "npm";
    PM["YARN"] = "yarn";
    PM["PNPM"] = "pnpm";
    PM["YARN_BERRY"] = "yarn-berry";
    PM["BUN"] = "bun";
})(PM || (exports.PM = PM = {}));
// Cache for resolved paths
const pmPathCache = {
    [PM.NPM]: undefined,
    [PM.YARN]: undefined,
    [PM.PNPM]: undefined,
    [PM.YARN_BERRY]: undefined,
    [PM.BUN]: undefined,
};
function resolveCommand(pm) {
    const fallback = pm === PM.YARN_BERRY ? "yarn" : pm;
    if (process.platform !== "win32") {
        return fallback;
    }
    try {
        return which.sync(fallback);
    }
    catch {
        // If `which` fails (not found), still return the fallback string
        return fallback;
    }
}
function getPackageManagerCommand(pm) {
    if (pmPathCache[pm] !== undefined) {
        return pmPathCache[pm];
    }
    const resolved = resolveCommand(pm);
    pmPathCache[pm] = resolved;
    return resolved;
}
function detectPackageManagerByEnv() {
    var _a;
    const packageJsonPath = path.join(process.cwd(), "package.json");
    const packageManager = fs.existsSync(packageJsonPath) ? (_a = JSON.parse(fs.readFileSync(packageJsonPath, "utf8"))) === null || _a === void 0 ? void 0 : _a.packageManager : undefined;
    const priorityChecklist = [
        (key) => { var _a; return (_a = process.env.npm_config_user_agent) === null || _a === void 0 ? void 0 : _a.includes(key); },
        (key) => { var _a; return (_a = process.env.npm_execpath) === null || _a === void 0 ? void 0 : _a.includes(key); },
        (key) => packageManager === null || packageManager === void 0 ? void 0 : packageManager.startsWith(`${key}@`),
    ];
    const pms = Object.values(PM).filter(pm => pm !== PM.YARN_BERRY);
    for (const checker of priorityChecklist) {
        for (const pm of pms) {
            if (checker(pm)) {
                return pm;
            }
        }
    }
    return null;
}
function detectPackageManagerByLockfile(cwd) {
    const has = (file) => fs.existsSync(path.join(cwd, file));
    const detected = [];
    if (has("yarn.lock")) {
        detected.push(PM.YARN);
    }
    if (has("pnpm-lock.yaml")) {
        detected.push(PM.PNPM);
    }
    if (has("package-lock.json")) {
        detected.push(PM.NPM);
    }
    if (has("bun.lock") || has("bun.lockb")) {
        detected.push(PM.BUN);
    }
    if (detected.length === 1) {
        return detected[0];
    }
    return null;
}
function detectYarnBerry() {
    try {
        const version = (0, child_process_1.execSync)("yarn --version").toString().trim();
        if (parseInt(version.split(".")[0]) > 1) {
            return PM.YARN_BERRY;
        }
    }
    catch (_e) {
        // If `yarn` is not found or another error occurs, fallback to the regular Yarn
    }
    return PM.YARN;
}
//# sourceMappingURL=packageManager.js.map