/*
 * Decompiled with CFR 0.152.
 */
package stirling.software.SPDF.controller.api.pipeline;

import com.fasterxml.jackson.databind.ObjectMapper;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.OutputStream;
import java.nio.charset.StandardCharsets;
import java.nio.file.CopyOption;
import java.nio.file.FileSystemException;
import java.nio.file.FileVisitor;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.StandardCopyOption;
import java.nio.file.attribute.FileAttribute;
import java.time.LocalDate;
import java.time.LocalTime;
import java.time.format.DateTimeFormatter;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Locale;
import java.util.Optional;
import java.util.stream.Stream;
import lombok.Generated;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.core.io.ByteArrayResource;
import org.springframework.core.io.Resource;
import org.springframework.scheduling.annotation.Scheduled;
import org.springframework.stereotype.Service;
import stirling.software.SPDF.controller.api.pipeline.PipelineProcessor;
import stirling.software.SPDF.model.PipelineConfig;
import stirling.software.SPDF.model.PipelineOperation;
import stirling.software.SPDF.model.PipelineResult;
import stirling.software.SPDF.service.ApiDocService;
import stirling.software.common.configuration.RuntimePathConfig;
import stirling.software.common.service.PostHogService;
import stirling.software.common.util.FileMonitor;

@Service
public class PipelineDirectoryProcessor {
    @Generated
    private static final Logger log = LoggerFactory.getLogger(PipelineDirectoryProcessor.class);
    private final ObjectMapper objectMapper;
    private final ApiDocService apiDocService;
    private final PipelineProcessor processor;
    private final FileMonitor fileMonitor;
    private final PostHogService postHogService;
    private final String watchedFoldersDir;
    private final String finishedFoldersDir;

    public PipelineDirectoryProcessor(ObjectMapper objectMapper, ApiDocService apiDocService, PipelineProcessor processor, FileMonitor fileMonitor, PostHogService postHogService, RuntimePathConfig runtimePathConfig) {
        this.objectMapper = objectMapper;
        this.apiDocService = apiDocService;
        this.processor = processor;
        this.fileMonitor = fileMonitor;
        this.postHogService = postHogService;
        this.watchedFoldersDir = runtimePathConfig.getPipelineWatchedFoldersPath();
        this.finishedFoldersDir = runtimePathConfig.getPipelineFinishedFoldersPath();
    }

    @Scheduled(fixedRate=60000L)
    public void scanFolders() {
        Path watchedFolderPath = Paths.get(this.watchedFoldersDir, new String[0]).toAbsolutePath();
        if (!Files.exists(watchedFolderPath, new LinkOption[0])) {
            try {
                Files.createDirectories(watchedFolderPath, new FileAttribute[0]);
                log.info("Created directory: {}", (Object)watchedFolderPath);
            }
            catch (IOException e) {
                log.error("Error creating directory: {}", (Object)watchedFolderPath, (Object)e);
                return;
            }
        }
        try {
            Files.walkFileTree(watchedFolderPath, (FileVisitor<? super Path>)new /* Unavailable Anonymous Inner Class!! */);
        }
        catch (IOException e) {
            log.error("Error walking through directory: {}", (Object)watchedFolderPath, (Object)e);
        }
    }

    public void handleDirectory(Path dir) throws IOException {
        log.info("Handling directory: {}", (Object)dir);
        Path processingDir = this.createProcessingDirectory(dir);
        Optional jsonFileOptional = this.findJsonFile(dir);
        if (jsonFileOptional.isEmpty()) {
            log.warn("No .JSON settings file found. No processing will happen for dir {}.", (Object)dir);
            return;
        }
        Path jsonFile = (Path)jsonFileOptional.get();
        PipelineConfig config = this.readAndParseJson(jsonFile);
        this.processPipelineOperations(dir, processingDir, jsonFile, config);
    }

    private Path createProcessingDirectory(Path dir) throws IOException {
        Path processingDir = dir.resolve("processing");
        if (!Files.exists(processingDir, new LinkOption[0])) {
            Files.createDirectory(processingDir, new FileAttribute[0]);
            log.info("Created processing directory: {}", (Object)processingDir);
        }
        return processingDir;
    }

    private Optional<Path> findJsonFile(Path dir) throws IOException {
        try (Stream<Path> paths = Files.list(dir);){
            Optional<Path> optional = paths.filter(file -> file.toString().endsWith(".json")).findFirst();
            return optional;
        }
    }

    private PipelineConfig readAndParseJson(Path jsonFile) throws IOException {
        String jsonString = new String(Files.readAllBytes(jsonFile), StandardCharsets.UTF_8);
        log.debug("Reading JSON file: {}", (Object)jsonFile);
        return (PipelineConfig)this.objectMapper.readValue(jsonString, PipelineConfig.class);
    }

    private void processPipelineOperations(Path dir, Path processingDir, Path jsonFile, PipelineConfig config) throws IOException {
        for (PipelineOperation operation : config.getOperations()) {
            this.validateOperation(operation);
            File[] files = this.collectFilesForProcessing(dir, jsonFile, operation);
            if (files.length == 0) {
                log.debug("No files detected for {} ", (Object)dir);
                return;
            }
            List<String> operationNames = config.getOperations().stream().map(PipelineOperation::getOperation).toList();
            HashMap<String, Object> properties = new HashMap<String, Object>();
            properties.put("operations", operationNames);
            properties.put("fileCount", files.length);
            this.postHogService.captureEvent("pipeline_directory_event", properties);
            List filesToProcess = this.prepareFilesForProcessing(files, processingDir);
            this.runPipelineAgainstFiles(filesToProcess, config, dir, processingDir);
        }
    }

    private void validateOperation(PipelineOperation operation) throws IOException {
        if (!this.apiDocService.isValidOperation(operation.getOperation(), operation.getParameters())) {
            throw new IOException("Invalid operation: " + operation.getOperation());
        }
    }

    private File[] collectFilesForProcessing(Path dir, Path jsonFile, PipelineOperation operation) throws IOException {
        List inputExtensions = this.apiDocService.getExtensionTypes(false, operation.getOperation());
        log.info("Allowed extensions for operation {}: {}", (Object)operation.getOperation(), (Object)inputExtensions);
        boolean allowAllFiles = inputExtensions.contains("ALL");
        try (Stream<Path> paths = Files.list(dir);){
            File[] files = (File[])paths.filter(path -> {
                boolean isAllowed;
                if (Files.isDirectory(path, new LinkOption[0])) {
                    return false;
                }
                if (path.equals(jsonFile)) {
                    return false;
                }
                String filename = path.getFileName().toString();
                String extension = filename.contains(".") ? filename.substring(filename.lastIndexOf(".") + 1).toLowerCase(Locale.ROOT) : "";
                boolean bl = isAllowed = allowAllFiles || inputExtensions.contains(extension.toLowerCase());
                if (!isAllowed) {
                    log.info("Skipping file with unsupported extension: {} ({})", (Object)filename, (Object)extension);
                }
                return isAllowed;
            }).map(Path::toAbsolutePath).filter(path -> {
                boolean isReady = this.fileMonitor.isFileReadyForProcessing(path);
                if (!isReady) {
                    log.info("File not ready for processing (locked/created last 5s): {}", path);
                }
                return isReady;
            }).map(Path::toFile).toArray(File[]::new);
            log.info("Collected {} files for processing for {}", (Object)files.length, (Object)dir.toAbsolutePath().toString());
            File[] fileArray = files;
            return fileArray;
        }
    }

    private List<File> prepareFilesForProcessing(File[] files, Path processingDir) throws IOException {
        ArrayList<File> filesToProcess = new ArrayList<File>();
        for (File file : files) {
            Path targetPath = this.resolveUniqueFilePath(processingDir, file.getName());
            int maxRetries = 3;
            int retryDelayMs = 500;
            boolean moved = false;
            for (int attempt = 1; attempt <= maxRetries; ++attempt) {
                try {
                    Files.move(file.toPath(), targetPath, StandardCopyOption.REPLACE_EXISTING);
                    moved = true;
                    break;
                }
                catch (FileSystemException e) {
                    if (attempt >= maxRetries) continue;
                    log.info("File move failed (attempt {}), retrying...", (Object)attempt);
                    try {
                        Thread.sleep(retryDelayMs * (int)Math.pow(2.0, attempt - 1));
                    }
                    catch (InterruptedException e1) {
                        log.error("prepareFilesForProcessing failure", (Throwable)e);
                    }
                    continue;
                }
            }
            if (moved) {
                filesToProcess.add(targetPath.toFile());
                continue;
            }
            log.error("Failed to move file after {} attempts: {}", (Object)maxRetries, (Object)file.getName());
        }
        return filesToProcess;
    }

    private Path resolveUniqueFilePath(Path directory, String originalFileName) {
        Path filePath = directory.resolve(originalFileName);
        int counter = 1;
        while (Files.exists(filePath, new LinkOption[0])) {
            String newName = this.appendSuffixToFileName(originalFileName, "(" + counter + ")");
            filePath = directory.resolve(newName);
            ++counter;
        }
        return filePath;
    }

    private String appendSuffixToFileName(String originalFileName, String suffix) {
        int dotIndex = originalFileName.lastIndexOf(46);
        if (dotIndex == -1) {
            return originalFileName + suffix;
        }
        return originalFileName.substring(0, dotIndex) + suffix + originalFileName.substring(dotIndex);
    }

    private void runPipelineAgainstFiles(List<File> filesToProcess, PipelineConfig config, Path dir, Path processingDir) throws IOException {
        try {
            List inputFiles = this.processor.generateInputFiles(filesToProcess.toArray(new File[0]));
            if (inputFiles == null || inputFiles.isEmpty()) {
                return;
            }
            PipelineResult result = this.processor.runPipelineAgainstFiles(inputFiles, config);
            if (result.isHasErrors()) {
                log.error("Errors occurred during processing, retaining original files");
                this.moveToErrorDirectory(filesToProcess, dir);
            } else {
                this.moveAndRenameFiles(result.getOutputFiles(), config, dir);
                this.deleteOriginalFiles(filesToProcess, processingDir);
            }
        }
        catch (Exception e) {
            log.error("Error during processing", (Throwable)e);
            this.moveFilesBack(filesToProcess, processingDir);
        }
    }

    private void moveToErrorDirectory(List<File> files, Path originalDir) throws IOException {
        Path errorDir = originalDir.resolve("error");
        if (!Files.exists(errorDir, new LinkOption[0])) {
            Files.createDirectories(errorDir, new FileAttribute[0]);
        }
        for (File file : files) {
            Path target = errorDir.resolve(file.getName());
            Files.move(file.toPath(), target, new CopyOption[0]);
            log.info("Moved failed file to error directory for investigation: {}", (Object)target);
        }
    }

    private void moveAndRenameFiles(List<Resource> resources, PipelineConfig config, Path dir) throws IOException {
        for (Resource resource : resources) {
            String outputFileName = this.createOutputFileName(resource, config);
            Path outputPath = this.determineOutputPath(config, dir);
            if (!Files.exists(outputPath, new LinkOption[0])) {
                Files.createDirectories(outputPath, new FileAttribute[0]);
                log.info("Created directory: {}", (Object)outputPath);
            }
            Path outputFile = outputPath.resolve(outputFileName);
            try (FileOutputStream os = new FileOutputStream(outputFile.toFile());){
                ((OutputStream)os).write(((ByteArrayResource)resource).getByteArray());
            }
            log.info("File moved and renamed to {}", (Object)outputFile);
        }
    }

    private String createOutputFileName(Resource resource, PipelineConfig config) {
        String resourceName = resource.getFilename();
        String baseName = resourceName.substring(0, resourceName.lastIndexOf(46));
        String extension = resourceName.substring(resourceName.lastIndexOf(46) + 1);
        String outputFileName = config.getOutputPattern().replace("{filename}", baseName).replace("{pipelineName}", config.getName()).replace("{date}", LocalDate.now().format(DateTimeFormatter.ofPattern("yyyyMMdd"))).replace("{time}", LocalTime.now().format(DateTimeFormatter.ofPattern("HHmmss"))) + "." + extension;
        return outputFileName;
    }

    private Path determineOutputPath(PipelineConfig config, Path dir) {
        String outputDir = config.getOutputDir().replace("{outputFolder}", this.finishedFoldersDir).replace("{folderName}", dir.toString()).replaceAll("\\\\?watchedFolders", "");
        return Paths.get(outputDir, new String[0]).isAbsolute() ? Paths.get(outputDir, new String[0]) : Paths.get(".", outputDir);
    }

    private void deleteOriginalFiles(List<File> filesToProcess, Path processingDir) throws IOException {
        for (File file : filesToProcess) {
            Files.deleteIfExists(processingDir.resolve(file.getName()));
            log.info("Deleted original file: {}", (Object)file.getName());
        }
    }

    private void moveFilesBack(List<File> filesToProcess, Path processingDir) {
        for (File file : filesToProcess) {
            try {
                Files.move(processingDir.resolve(file.getName()), file.toPath(), new CopyOption[0]);
                log.info("Moved file back to original location: {} , {}", (Object)file.toPath(), (Object)file.getName());
            }
            catch (IOException e) {
                log.error("Error moving file back to original location: {}", (Object)file.getName(), (Object)e);
            }
        }
    }
}

