/*
 * Decompiled with CFR 0.152.
 */
package stirling.software.common.controller;

import io.swagger.v3.oas.annotations.Operation;
import io.swagger.v3.oas.annotations.tags.Tag;
import jakarta.servlet.http.HttpServletRequest;
import java.net.URLEncoder;
import java.nio.charset.StandardCharsets;
import java.util.List;
import java.util.Map;
import lombok.Generated;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.DeleteMapping;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RestController;
import stirling.software.common.model.job.JobResult;
import stirling.software.common.model.job.ResultFile;
import stirling.software.common.service.FileStorage;
import stirling.software.common.service.JobOwnershipService;
import stirling.software.common.service.JobQueue;
import stirling.software.common.service.TaskManager;
import stirling.software.common.util.RegexPatternUtils;

@RestController
@RequestMapping(value={"/api/v1/general"})
@Tag(name="Job Management", description="Job Management API")
public class JobController {
    @Generated
    private static final Logger log = LoggerFactory.getLogger(JobController.class);
    private final TaskManager taskManager;
    private final FileStorage fileStorage;
    private final JobQueue jobQueue;
    private final HttpServletRequest request;
    @Autowired(required=false)
    private JobOwnershipService jobOwnershipService;

    @GetMapping(value={"/job/{jobId}"})
    @Operation(summary="Get job status")
    public ResponseEntity<?> getJobStatus(@PathVariable(value="jobId") String jobId) {
        if (!this.validateJobAccess(jobId)) {
            log.warn("Unauthorized attempt to access job status: {}", (Object)jobId);
            return ResponseEntity.status((int)403).body(Map.of("message", "You are not authorized to access this job"));
        }
        JobResult result = this.taskManager.getJobResult(jobId);
        if (result == null) {
            return ResponseEntity.notFound().build();
        }
        if (!result.isComplete() && this.jobQueue.isJobQueued(jobId)) {
            int position = this.jobQueue.getJobPosition(jobId);
            Map<String, Map<String, Integer>> resultWithQueueInfo = Map.of("jobResult", result, "queueInfo", Map.of("inQueue", true, "position", position));
            return ResponseEntity.ok(resultWithQueueInfo);
        }
        return ResponseEntity.ok((Object)result);
    }

    @GetMapping(value={"/job/{jobId}/result"})
    @Operation(summary="Get job result")
    public ResponseEntity<?> getJobResult(@PathVariable(value="jobId") String jobId) {
        if (!this.validateJobAccess(jobId)) {
            log.warn("Unauthorized attempt to access job result: {}", (Object)jobId);
            return ResponseEntity.status((int)403).body(Map.of("message", "You are not authorized to access this job"));
        }
        JobResult result = this.taskManager.getJobResult(jobId);
        if (result == null) {
            return ResponseEntity.notFound().build();
        }
        if (!result.isComplete()) {
            return ResponseEntity.badRequest().body((Object)"Job is not complete yet");
        }
        if (result.getError() != null) {
            return ResponseEntity.badRequest().body((Object)("Job failed: " + result.getError()));
        }
        if (result.hasMultipleFiles()) {
            return ResponseEntity.ok().contentType(MediaType.APPLICATION_JSON).body(Map.of("jobId", jobId, "hasMultipleFiles", true, "files", result.getAllResultFiles()));
        }
        if (result.hasFiles() && !result.hasMultipleFiles()) {
            try {
                List files = result.getAllResultFiles();
                ResultFile singleFile = (ResultFile)files.get(0);
                byte[] fileContent = this.fileStorage.retrieveBytes(singleFile.getFileId());
                return ((ResponseEntity.BodyBuilder)((ResponseEntity.BodyBuilder)ResponseEntity.ok().header("Content-Type", new String[]{singleFile.getContentType()})).header("Content-Disposition", new String[]{this.createContentDispositionHeader(singleFile.getFileName())})).body((Object)fileContent);
            }
            catch (Exception e) {
                log.error("Error retrieving file for job {}: {}", new Object[]{jobId, e.getMessage(), e});
                return ResponseEntity.internalServerError().body((Object)("Error retrieving file: " + e.getMessage()));
            }
        }
        return ResponseEntity.ok((Object)result.getResult());
    }

    @DeleteMapping(value={"/job/{jobId}"})
    @Operation(summary="Cancel a job")
    public ResponseEntity<?> cancelJob(@PathVariable(value="jobId") String jobId) {
        JobResult result;
        log.debug("Request to cancel job: {}", (Object)jobId);
        if (!this.validateJobAccess(jobId)) {
            log.warn("Unauthorized attempt to cancel job: {}", (Object)jobId);
            return ResponseEntity.status((int)403).body(Map.of("message", "You are not authorized to cancel this job"));
        }
        boolean cancelled = false;
        int queuePosition = -1;
        if (this.jobQueue.isJobQueued(jobId)) {
            queuePosition = this.jobQueue.getJobPosition(jobId);
            cancelled = this.jobQueue.cancelJob(jobId);
            log.info("Cancelled queued job: {} (was at position {})", (Object)jobId, (Object)queuePosition);
        }
        if (!cancelled && (result = this.taskManager.getJobResult(jobId)) != null && !result.isComplete()) {
            this.taskManager.setError(jobId, "Job was cancelled by user");
            cancelled = true;
            log.info("Marked job as cancelled in TaskManager: {}", (Object)jobId);
        }
        if (cancelled) {
            return ResponseEntity.ok(Map.of("message", "Job cancelled successfully", "wasQueued", queuePosition >= 0, "queuePosition", queuePosition >= 0 ? Integer.valueOf(queuePosition) : "n/a"));
        }
        result = this.taskManager.getJobResult(jobId);
        if (result == null) {
            return ResponseEntity.notFound().build();
        }
        if (result.isComplete()) {
            return ResponseEntity.badRequest().body(Map.of("message", "Cannot cancel job that is already complete"));
        }
        return ResponseEntity.internalServerError().body(Map.of("message", "Failed to cancel job for unknown reason"));
    }

    @GetMapping(value={"/job/{jobId}/result/files"})
    @Operation(summary="Get job result files")
    public ResponseEntity<?> getJobFiles(@PathVariable(value="jobId") String jobId) {
        if (!this.validateJobAccess(jobId)) {
            log.warn("Unauthorized attempt to access job files: {}", (Object)jobId);
            return ResponseEntity.status((int)403).body(Map.of("message", "You are not authorized to access this job"));
        }
        JobResult result = this.taskManager.getJobResult(jobId);
        if (result == null) {
            return ResponseEntity.notFound().build();
        }
        if (!result.isComplete()) {
            return ResponseEntity.badRequest().body((Object)"Job is not complete yet");
        }
        if (result.getError() != null) {
            return ResponseEntity.badRequest().body((Object)("Job failed: " + result.getError()));
        }
        List files = result.getAllResultFiles();
        return ResponseEntity.ok(Map.of("jobId", jobId, "fileCount", files.size(), "files", files));
    }

    @GetMapping(value={"/files/{fileId}/metadata"})
    @Operation(summary="Get file metadata")
    public ResponseEntity<?> getFileMetadata(@PathVariable(value="fileId") String fileId) {
        try {
            if (!this.fileStorage.fileExists(fileId)) {
                return ResponseEntity.notFound().build();
            }
            ResultFile resultFile = this.taskManager.findResultFileByFileId(fileId);
            if (resultFile != null) {
                return ResponseEntity.ok((Object)resultFile);
            }
            long fileSize = this.fileStorage.getFileSize(fileId);
            return ResponseEntity.ok(Map.of("fileId", fileId, "fileName", "unknown", "contentType", "application/octet-stream", "fileSize", fileSize));
        }
        catch (Exception e) {
            log.error("Error retrieving file metadata {}: {}", new Object[]{fileId, e.getMessage(), e});
            return ResponseEntity.internalServerError().body((Object)("Error retrieving file metadata: " + e.getMessage()));
        }
    }

    @GetMapping(value={"/files/{fileId}"})
    @Operation(summary="Download a file")
    public ResponseEntity<?> downloadFile(@PathVariable(value="fileId") String fileId) {
        try {
            if (!this.fileStorage.fileExists(fileId)) {
                return ResponseEntity.notFound().build();
            }
            byte[] fileContent = this.fileStorage.retrieveBytes(fileId);
            ResultFile resultFile = this.taskManager.findResultFileByFileId(fileId);
            String fileName = resultFile != null ? resultFile.getFileName() : "download";
            String contentType = resultFile != null ? resultFile.getContentType() : "application/octet-stream";
            return ((ResponseEntity.BodyBuilder)((ResponseEntity.BodyBuilder)ResponseEntity.ok().header("Content-Type", new String[]{contentType})).header("Content-Disposition", new String[]{this.createContentDispositionHeader(fileName)})).body((Object)fileContent);
        }
        catch (Exception e) {
            log.error("Error retrieving file {}: {}", new Object[]{fileId, e.getMessage(), e});
            return ResponseEntity.internalServerError().body((Object)("Error retrieving file: " + e.getMessage()));
        }
    }

    private String createContentDispositionHeader(String fileName) {
        try {
            String encodedFileName = RegexPatternUtils.getInstance().getPlusSignPattern().matcher(URLEncoder.encode(fileName, StandardCharsets.UTF_8)).replaceAll("%20");
            return "attachment; filename=\"" + fileName + "\"; filename*=UTF-8''" + encodedFileName;
        }
        catch (Exception e) {
            return "attachment; filename=\"" + fileName + "\"";
        }
    }

    private boolean validateJobAccess(String jobId) {
        if (this.jobOwnershipService != null) {
            try {
                return this.jobOwnershipService.validateJobAccess(jobId);
            }
            catch (SecurityException e) {
                log.warn("Job ownership validation failed for jobId {}: {}", (Object)jobId, (Object)e.getMessage());
                return false;
            }
        }
        return true;
    }

    @Generated
    public JobController(TaskManager taskManager, FileStorage fileStorage, JobQueue jobQueue, HttpServletRequest request) {
        this.taskManager = taskManager;
        this.fileStorage = fileStorage;
        this.jobQueue = jobQueue;
        this.request = request;
    }
}

