use chunker::sizers::PretokenizerSizer;
use chunker::{Chunk, File, FileChunker, SizeChunker, SplitCodeChunker};
use std::ffi::CString;
use std::os::raw::{c_char, c_int, c_void};
use std::ptr;
use std::slice;

#[repr(C)]
pub struct CChunker {
    _private: [u8; 0],
}

#[repr(C)]
#[derive(Copy, Clone, Debug)]
pub struct CStringSlice {
    pub ptr: *const c_char,
    pub len: usize,
}

#[repr(C)]
pub struct CChunk {
    pub start_byte: usize,
    pub end_byte: usize,
    pub length: usize,           // Calculated as end_byte - start_byte
    pub start_line: usize,       // Line number where chunk starts (1-indexed)
    pub content: CStringSlice,   // Points directly into source code
    pub file_path: CStringSlice, // Points directly into file_path input
    pub language: CStringSlice,
}

#[repr(C)]
#[derive(Copy, Clone, Debug)]
pub struct CFile {
    pub source_code: CStringSlice,
    pub file_path: CStringSlice,
}

#[repr(C)]
pub struct CChunkArray {
    _private: [u8; 0],
}

// Wrapper to hold fat pointer to the trait object
struct ChunkerWrapper {
    inner: Box<dyn FileChunker>,
}

// Error codes
/// Success return code
pub const CHUNKER_SUCCESS: c_int = 0;
/// Null pointer error
pub const CHUNKER_ERROR_NULL_POINTER: c_int = -1;
/// Invalid string error
pub const CHUNKER_ERROR_INVALID_STRING: c_int = -2;
/// Chunking operation failed
pub const CHUNKER_ERROR_CHUNKING_FAILED: c_int = -3;

// Language IDs
pub const LANGUAGE_NONE: c_int = -1;

/// Creates a new SplitCodeChunker with fallback
///
/// # Safety
/// This function is safe to call from C as long as the returned pointer
/// is eventually freed with `chunker_free()`.
#[unsafe(no_mangle)]
pub unsafe extern "C" fn chunker_new_split_code(max_chunk_size: usize) -> *mut CChunker {
    if max_chunk_size == 0 {
        return ptr::null_mut();
    }

    let fallback = match SizeChunker::new(max_chunk_size, 0) {
        Ok(chunker) => Box::new(chunker),
        Err(_) => return ptr::null_mut(),
    };

    let chunker = SplitCodeChunker::new(max_chunk_size).with_fallback(fallback);

    let wrapper = Box::new(ChunkerWrapper {
        inner: Box::new(chunker),
    });
    Box::into_raw(wrapper) as *mut CChunker
}

/// Creates a new SplitCodeChunker with fallback using BERT pre-tokenization token limits.
///
/// # Safety
/// This function is safe to call from C as long as the returned pointer
/// is eventually freed with `chunker_free()`.
#[unsafe(no_mangle)]
pub unsafe extern "C" fn chunker_new_split_code_with_pre_bert(max_tokens: usize) -> *mut CChunker {
    if max_tokens == 0 {
        return ptr::null_mut();
    }

    let sizer = match PretokenizerSizer::with_bert_pre(max_tokens) {
        Ok(sizer) => Box::new(sizer),
        Err(_) => return ptr::null_mut(),
    };
    let fallback = Box::new(SizeChunker::new_sizer(sizer.clone(), 0));
    let chunker = SplitCodeChunker::new_sizer(sizer).with_fallback(fallback);

    let wrapper = Box::new(ChunkerWrapper {
        inner: Box::new(chunker),
    });
    Box::into_raw(wrapper) as *mut CChunker
}

/// Creates a new SizeChunker
///
/// # Safety
/// This function is safe to call from C as long as the returned pointer
/// is eventually freed with `chunker_free()`.
#[unsafe(no_mangle)]
pub unsafe extern "C" fn chunker_new_size(
    max_chunk_size: usize,
    chunk_overlap: usize,
) -> *mut CChunker {
    if max_chunk_size == 0 {
        return ptr::null_mut();
    }

    let chunker = match SizeChunker::new(max_chunk_size, chunk_overlap) {
        Ok(chunker) => chunker,
        Err(_) => return ptr::null_mut(),
    };

    let wrapper = Box::new(ChunkerWrapper {
        inner: Box::new(chunker),
    });
    Box::into_raw(wrapper) as *mut CChunker
}

/// Creates a new SplitCodeChunker without fallback
///
/// # Safety
/// This function is safe to call from C as long as the returned pointer
/// is eventually freed with `chunker_free()`.
#[unsafe(no_mangle)]
pub unsafe extern "C" fn chunker_new_split_code_simple(max_chunk_size: usize) -> *mut CChunker {
    if max_chunk_size == 0 {
        return ptr::null_mut();
    }

    let chunker = SplitCodeChunker::new(max_chunk_size);

    let wrapper = Box::new(ChunkerWrapper {
        inner: Box::new(chunker),
    });
    Box::into_raw(wrapper) as *mut CChunker
}

/// Frees the chunker and associated memory
///
/// # Safety
/// The caller must ensure that:
/// - `chunker` is a valid pointer returned from a `chunker_new_*` function
/// - `chunker` is not used after this call
/// - This function is called exactly once per chunker
#[unsafe(no_mangle)]
pub unsafe extern "C" fn chunker_free(chunker: *mut CChunker) {
    if !chunker.is_null() {
        unsafe {
            let _ = Box::from_raw(chunker as *mut ChunkerWrapper);
        }
    }
}

/// Chunks multiple files and appends results to the provided chunk array
/// Returns error code (CHUNKER_SUCCESS on success)
///
/// # Safety
/// The caller must ensure that:
/// - `chunker` is a valid pointer returned from a `chunker_new_*` function
/// - `files` points to a valid array of `files_count` CFile structures
/// - `chunk_array` is a valid pointer returned from `chunker_create_chunk_array`
/// - The source code and file path data in `files` remains valid until the chunks are no longer needed
/// - All pointers are properly aligned
#[unsafe(no_mangle)]
pub unsafe extern "C" fn chunker_chunk_files(
    chunker: *mut CChunker,
    files: *const CFile,
    files_count: usize,
    chunks: *mut CChunkArray,
    error: *mut CStringSlice,
) -> c_int {
    if chunker.is_null() || files.is_null() || chunks.is_null() || files_count == 0 {
        return CHUNKER_ERROR_NULL_POINTER;
    }

    unsafe {
        let chunker_wrapper = &*(chunker as *mut ChunkerWrapper);
        let chunk_array = &mut *(chunks as *mut Vec<Chunk>);

        let file_slice = slice::from_raw_parts(files, files_count);

        for file in file_slice {
            let source_code_slice =
                slice::from_raw_parts(file.source_code.ptr as *const u8, file.source_code.len);
            let source_code = match std::str::from_utf8(source_code_slice) {
                Ok(s) => s,
                Err(_) => return CHUNKER_ERROR_INVALID_STRING,
            };

            let file_path_slice =
                slice::from_raw_parts(file.file_path.ptr as *const u8, file.file_path.len);
            let file_path = match std::str::from_utf8(file_path_slice) {
                Ok(s) => s,
                Err(_) => return CHUNKER_ERROR_INVALID_STRING,
            };

            let file = File {
                file_path,
                source_code,
            };

            let result = chunker_wrapper.inner.chunk_file(chunk_array, &file);
            if result.is_err() {
                let error_message = match result {
                    Err(e) => e.to_string(),
                    _ => String::from("Unknown error"),
                };

                let c_error = CString::new(error_message).unwrap();
                let len = c_error.as_bytes().len();

                *error = CStringSlice {
                    ptr: c_error.into_raw(),
                    len,
                };

                return CHUNKER_ERROR_CHUNKING_FAILED;
            }
        }

        CHUNKER_SUCCESS
    }
}

/// Creates a new empty chunk array
///
/// # Safety
/// This function is safe to call from C as long as the returned pointer
/// is eventually freed with `chunker_free_chunks()`.
#[unsafe(no_mangle)]
pub unsafe extern "C" fn chunker_create_chunk_array() -> *mut CChunkArray {
    let chunks = Box::new(Vec::<Chunk>::new());

    Box::into_raw(chunks) as *mut CChunkArray
}

/// Clears all chunks from the array but keeps the allocated memory for reuse
///
/// # Safety
/// The caller must ensure that `chunks` is either null or a valid pointer
/// returned from `chunker_create_chunk_array`.
#[unsafe(no_mangle)]
pub unsafe extern "C" fn chunker_clear_chunks(chunks: *mut CChunkArray) {
    if chunks.is_null() {
        return;
    }

    unsafe {
        let chunk_array = &mut *(chunks as *mut Vec<Chunk>);

        chunk_array.clear();
    }
}

type CChunkYieldFunc = extern "C" fn(item: *const CChunk, data: *const c_void) -> c_int;

/// Enumerates through chunks in a chunk array using a Go-style yield function pattern.
///
/// This function iterates through each chunk in the provided chunk array and calls
/// the yield function for each chunk. The enumeration can be terminated early if
/// the yield function returns a non-zero value.
///
/// # Parameters
/// - `chunks`: A mutable pointer to a CChunkArray (Vec<Chunk>) to enumerate
/// - `data`: A pointer to user data that will be passed to the yield function
/// - `yield_fn`: A function pointer that will be called for each chunk. Should return
///   0 to continue enumeration or non-zero to stop
///
/// # Safety
/// - `chunks` must be a valid pointer returned from `chunker_create_chunk_array` or null
/// - `yield_fn` must be a valid function pointer that doesn't store chunk data pointers
#[unsafe(no_mangle)]
pub unsafe extern "C" fn chunker_enumerate_chunks(
    chunks: *mut CChunkArray,
    data: *const c_void,
    yield_fn: CChunkYieldFunc,
) {
    if chunks.is_null() {
        return;
    }

    unsafe {
        let chunk_array = &mut *(chunks as *mut Vec<Chunk>);

        for chunk in chunk_array {
            let result = yield_fn(
                &CChunk {
                    start_byte: chunk.start_byte,
                    end_byte: chunk.end_byte,
                    length: chunk.end_byte - chunk.start_byte,
                    start_line: chunk.start_line,
                    content: CStringSlice {
                        ptr: chunk.content.as_ptr() as *const c_char,
                        len: chunk.content.len(),
                    },
                    file_path: CStringSlice {
                        ptr: chunk.file_path.as_ptr() as *const c_char,
                        len: chunk.file_path.len(),
                    },
                    language: CStringSlice {
                        ptr: chunk.language.as_ptr() as *const c_char,
                        len: chunk.language.len(),
                    },
                },
                data,
            );
            if result != 0 {
                break;
            }
        }
    }
}

/// Frees the chunk array and associated memory
/// Note: This does NOT free the source_code or file_path
///
/// # Safety
/// The caller must ensure that:
/// - `chunks` is either null or a valid pointer returned from `chunker_create_chunk_array`
/// - `chunks` is not used after this call
/// - This function is called exactly once per chunk array
#[unsafe(no_mangle)]
pub unsafe extern "C" fn chunker_free_chunks(chunks: *mut CChunkArray) {
    if chunks.is_null() {
        return;
    }

    unsafe {
        let _ = Box::from_raw(chunks as *mut Vec<Chunk>);
    }
}
