use parser_core::{
    AnalysisResult,
    references::{ReferenceInfo, ReferenceTarget},
};

#[derive(Clone, Copy, Default, Debug, serde::Serialize)]
pub struct Stats {
    pub definitions: usize,
    pub resolved_references: usize,
    pub ambiguous_references: usize,
    pub unresolved_references: usize,
    pub imports: usize,
}

// Local trait to extract the target from a reference type
pub trait HasReferenceTarget {
    type Target;
    fn target(&self) -> &ReferenceTarget<Self::Target>;
}

impl<TargetResolutionType, ReferenceType, Metadata, FqnType> HasReferenceTarget
    for ReferenceInfo<TargetResolutionType, ReferenceType, Metadata, FqnType>
{
    type Target = TargetResolutionType;
    fn target(&self) -> &ReferenceTarget<Self::Target> {
        &self.target
    }
}

impl Stats {
    pub fn from_analysis_result<Fqn, DefinitionType, ImportType, RefType>(
        analysis_result: &AnalysisResult<Fqn, DefinitionType, ImportType, RefType>,
    ) -> Self
    where
        RefType: HasReferenceTarget,
    {
        let (resolved, ambiguous, unresolved) = analysis_result.references.iter().fold(
            (0usize, 0usize, 0usize),
            |(res, amb, unres), r| match r.target() {
                ReferenceTarget::Resolved(_) => (res + 1, amb, unres),
                ReferenceTarget::Ambiguous(_) => (res, amb + 1, unres),
                ReferenceTarget::Unresolved() => (res, amb, unres + 1),
            },
        );

        Stats {
            definitions: analysis_result.definitions.len(),
            imports: analysis_result.imports.len(),
            resolved_references: resolved,
            ambiguous_references: ambiguous,
            unresolved_references: unresolved,
        }
    }

    pub fn from_analysis_result_simple<Fqn, DefinitionType, ImportType>(
        analysis_result: &AnalysisResult<Fqn, DefinitionType, ImportType, ()>,
    ) -> Self {
        Stats {
            definitions: analysis_result.definitions.len(),
            imports: analysis_result.imports.len(),
            resolved_references: 0,
            ambiguous_references: 0,
            unresolved_references: 0,
        }
    }

    pub fn from_destructured_analysis_result<DefinitionType, ImportType, ReferenceType>(
        definitions: &[DefinitionType],
        imports: &[ImportType],
        references: &[ReferenceType],
    ) -> Self {
        Stats {
            definitions: definitions.len(),
            imports: imports.len(),
            resolved_references: 0,
            ambiguous_references: 0,
            unresolved_references: references.len(),
        }
    }

    pub fn to_json(self) -> Result<String, serde_json::Error> {
        serde_json::to_string_pretty(&self)
    }
}

impl std::ops::Add for Stats {
    type Output = Self;

    fn add(self, rhs: Self) -> Self::Output {
        Self {
            definitions: self.definitions + rhs.definitions,
            resolved_references: self.resolved_references + rhs.resolved_references,
            ambiguous_references: self.ambiguous_references + rhs.ambiguous_references,
            unresolved_references: self.unresolved_references + rhs.unresolved_references,
            imports: self.imports + rhs.imports,
        }
    }
}

impl std::ops::AddAssign for Stats {
    fn add_assign(&mut self, rhs: Self) {
        *self = *self + rhs;
    }
}

impl std::iter::Sum for Stats {
    fn sum<I: Iterator<Item = Self>>(iter: I) -> Self {
        iter.fold(Self::default(), |acc, s| acc + s)
    }
}
