package chunker

/*
#include <stdint.h>
#include "parser-c-bindings.h"

extern int goChunkerYieldFunc(CChunk*, void*);
*/
import "C"
import (
	"fmt"
	"iter"
	"runtime"
	"runtime/cgo"
	"unsafe"
)

// Chunk represents a code chunk
type Chunk struct {
	StartByte int
	EndByte   int
	Length    int
	StartLine int
	Content   string
	FilePath  string
	Language  string
}

// Chunker wraps the C chunker
//
// Chunker must have Close called to release memory.
type Chunker struct {
	pinner  runtime.Pinner
	files   []C.CFile
	chunks  *C.CChunkArray
	chunker *C.CChunker
}

func NewChunkerSize(maxChunkSize, chunkOverlap int) (*Chunker, error) {
	chunks := C.chunker_create_chunk_array()
	if chunks == nil {
		return nil, fmt.Errorf("failed to create chunk array")
	}
	chunker := C.chunker_new_size(C.size_t(maxChunkSize), C.size_t(chunkOverlap))
	if chunker == nil {
		return nil, fmt.Errorf("failed to create chunker")
	}
	return &Chunker{
		chunks:  chunks,
		chunker: chunker,
		files:   []C.CFile{},
	}, nil
}

func NewChunkerSplitCode(maxChunkSize int) (*Chunker, error) {
	chunks := C.chunker_create_chunk_array()
	if chunks == nil {
		return nil, fmt.Errorf("failed to create chunk array")
	}
	chunker := C.chunker_new_split_code(C.size_t(maxChunkSize))
	if chunker == nil {
		return nil, fmt.Errorf("failed to create chunker")
	}
	return &Chunker{
		chunks:  chunks,
		chunker: chunker,
		files:   []C.CFile{},
	}, nil
}

func NewChunkerSplitCodePreBert(maxTokens int) (*Chunker, error) {
	chunks := C.chunker_create_chunk_array()
	if chunks == nil {
		return nil, fmt.Errorf("failed to create chunk array")
	}
	chunker := C.chunker_new_split_code_with_pre_bert(C.size_t(maxTokens))
	if chunker == nil {
		return nil, fmt.Errorf("failed to create chunker")
	}
	return &Chunker{
		chunks:  chunks,
		chunker: chunker,
		files:   []C.CFile{},
	}, nil
}

func NewChunkerSplitCodeSimple(maxChunkSize int) (*Chunker, error) {
	chunks := C.chunker_create_chunk_array()
	if chunks == nil {
		return nil, fmt.Errorf("failed to create chunk array")
	}
	chunker := C.chunker_new_split_code_simple(C.size_t(maxChunkSize))
	if chunker == nil {
		return nil, fmt.Errorf("failed to create chunker")
	}
	return &Chunker{
		chunks:  chunks,
		chunker: chunker,
		files:   []C.CFile{},
	}, nil
}

// Close frees all chunker memory including files and chunks
func (c *Chunker) Close() {
	C.chunker_free_chunks(c.chunks)
	C.chunker_free(c.chunker)
	c.pinner.Unpin()
}

// Clear removes all files and chunks but keeps allocated memory for reuse
func (c *Chunker) Clear() {
	clear(c.files)
	c.files = c.files[:0]
	C.chunker_clear_chunks(c.chunks)
	c.pinner.Unpin()
}

func (c *Chunker) AddFile(filePath, sourceCode string) {
	cFilePath := goToStringSlice(filePath)
	cSourceCode := goToStringSlice(sourceCode)

	c.pinner.Pin(cFilePath.ptr)
	c.pinner.Pin(cSourceCode.ptr)

	c.files = append(c.files, C.CFile{
		file_path:   cFilePath,
		source_code: cSourceCode,
	})
}

func (c *Chunker) ChunkFiles() error {
	if len(c.files) == 0 {
		return nil
	}

	var errStringSlice C.CStringSlice
	result := C.chunker_chunk_files(
		c.chunker,
		&c.files[0],
		C.size_t(len(c.files)),
		c.chunks,
		&errStringSlice,
	)

	if result != C.CHUNKER_SUCCESS {
		return fmt.Errorf("chunking failed with status %d: %s", result, stringSliceToGo(errStringSlice))
	}

	return nil
}

//export goChunkerYieldFunc
func goChunkerYieldFunc(chunk *C.CChunk, data unsafe.Pointer) C.int {
	h := *(*cgo.Handle)(data)
	yield := h.Value().(func(c Chunk) bool)

	ok := yield(Chunk{
		FilePath:  stringSliceToGo(chunk.file_path),
		StartByte: int(chunk.start_byte),
		EndByte:   int(chunk.end_byte),
		Length:    int(chunk.length),
		StartLine: int(chunk.start_line),
		Content:   stringSliceToGo(chunk.content),
		Language:  stringSliceToGo(chunk.language),
	})

	if !ok {
		return -1
	}
	return 0
}

// Chunks returns an iterator of all chunks found.
func (c *Chunker) Chunks() iter.Seq[Chunk] {
	return func(yield func(c Chunk) bool) {
		h := cgo.NewHandle(yield)
		C.chunker_enumerate_chunks(c.chunks, unsafe.Pointer(&h), (C.CChunkYieldFunc)(C.goChunkerYieldFunc))
		h.Delete()
	}
}

// stringSliceToGo converts a C.CStringSlice to a Go string with zero copy
func stringSliceToGo(slice C.CStringSlice) string {
	if slice.ptr == nil || slice.len == 0 {
		return ""
	}
	return unsafe.String((*byte)(unsafe.Pointer(slice.ptr)), slice.len)
}

// goToStringSlice converts a Go string to a C.CStringSlice with zero copy
func goToStringSlice(str string) C.CStringSlice {
	return C.CStringSlice{
		ptr: (*C.char)(unsafe.Pointer(unsafe.StringData(str))),
		len: (C.size_t)(len(str)),
	}
}
