/**
 *    Copyright (C) 2023-present MongoDB, Inc.
 *
 *    This program is free software: you can redistribute it and/or modify
 *    it under the terms of the Server Side Public License, version 1,
 *    as published by MongoDB, Inc.
 *
 *    This program is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    Server Side Public License for more details.
 *
 *    You should have received a copy of the Server Side Public License
 *    along with this program. If not, see
 *    <http://www.mongodb.com/licensing/server-side-public-license>.
 *
 *    As a special exception, the copyright holders give permission to link the
 *    code of portions of this program with the OpenSSL library under certain
 *    conditions as described in each individual source file and distribute
 *    linked combinations including the program with the OpenSSL library. You
 *    must comply with the Server Side Public License in all respects for
 *    all of the code used other than as permitted herein. If you modify file(s)
 *    with this exception, you may extend this exception to your version of the
 *    file(s), but you are not obligated to do so. If you do not wish to do so,
 *    delete this exception statement from your version. If you delete this
 *    exception statement from all source files in the program, then also delete
 *    it in the license file.
 */


#pragma once

#include <vector>

#include "mongo/db/query/optimizer/bool_expression.h"
#include "mongo/util/str.h"

namespace mongo::optimizer {

/**
 * Prints a bool expr tree. Used for debugging.
 */
template <class T>
class BoolExprPrinter {
public:
    using BoolExprType = BoolExpr<T>;

    std::string transport(const typename BoolExprType::Atom& node) {
        return str::stream() << node.getExpr();
    }

    std::string printChildren(const std::string& separator,
                              const std::vector<std::string>& childResults) {
        str::stream os;
        os << "(";
        bool first = true;
        for (const std::string& child : childResults) {
            if (first) {
                first = false;
            } else {
                os << " " << separator << " ";
            }
            os << child;
        }
        os << ")";
        return os;
    }

    std::string transport(const typename BoolExprType::Conjunction& node,
                          std::vector<std::string> childResults) {
        return printChildren("^", childResults);
    }

    std::string transport(const typename BoolExprType::Disjunction& node,
                          std::vector<std::string> childResults) {
        return printChildren("U", childResults);
    }

    std::string print(const typename BoolExprType::Node& n) {
        return algebra::transport<false>(n, *this);
    }
};

}  // namespace mongo::optimizer
