'use strict';

/**
 * Performs concurrent upsert and delete operations against a small set of documents with a unique
 * index in place. One specific scenario this test exercises is upsert retry in the case where an
 * upsert generates an insert, which then fails due to another operation inserting first. This test
 * also involves collation.
 */
var $config = (function() {
    const data = {
        uniqueKeys: ["aa", "ab", "ba", "bb", "Aa", "Ab", "Ba", "BB"],
        getUniqueKey: function() {
            return this.uniqueKeys[Random.randInt(this.uniqueKeys.length)];
        },
    };

    const states = {
        delete: function(db, collName) {
            assert.commandWorked(
                db[collName].remove({uniqueKey: this.getUniqueKey()}, {justOne: true}));
        },
        upsert: function(db, collName) {
            const uniqueKey = this.getUniqueKey();
            const cmdRes = db.runCommand({
                update: collName,
                updates: [{
                    q: {uniqueKey: uniqueKey},
                    u: {$inc: {updates: 1}},
                    upsert: true,
                    collation: {locale: "en", strength: 2},
                }]
            });
            assert.commandWorked(cmdRes);
        },
    };

    const transitions = {
        upsert: {upsert: 0.5, delete: 0.5},
        delete: {upsert: 0.5, delete: 0.5},
    };

    function setup(db, collName, cluster) {
        db[collName].drop();
        assert.commandWorked(db[collName].createIndex(
            {uniqueKey: 1}, {unique: 1, collation: {locale: "en", strength: 2}}));
    }

    return {
        threadCount: 20,
        iterations: 100,
        states: states,
        startState: 'upsert',
        transitions: transitions,
        data: data,
        setup: setup,
    };
})();
