///////////////////////////////////////////////////////////////////////////////
//                                                         
// Matcher.cc
// ----------
// Regular Expression handling
//                                               
// Design and Implementation by Bjoern Lemke               
//                                                         
// (C)opyright 2000-2016 Bjoern Lemke
//
// IMPLEMENTATION MODULE
//
// Class: Matcher
// 
// Description: Regular expression utility class
//
// Status: CLEAN
//
///////////////////////////////////////////////////////////////////////////////

// POSIX INCLUDES
#include <stdlib.h>
// LFC INCLUDES
#include "Exception.h"
#include "Chain.h"
#include "Matcher.h"

#include <regex.h>

Matcher::Matcher(const Chain& expr)
{    
    _expr = expr;
    _isPrepared = false;
    _pRE = 0;
}

Matcher::~Matcher()
{
    if ( _pRE )
    {
	regfree((regex_t*)_pRE);
	free(_pRE);
    }
}

const Chain& Matcher::getExpr() const
{
    return _expr;
}

void Matcher::setExpr(const Chain& expr)
{
    _expr = expr;
}

void Matcher::prepare(bool isExtended )
{

    if ( _isPrepared && _pRE )
    {
	regfree((regex_t*)_pRE);
	free(_pRE);
	_pRE = 0;
    }
    
    if ( (char*)_expr != 0 )
    {
	_pRE = (regex_t*)malloc(sizeof(regex_t));
        int flag = 0;
        if ( isExtended )
           flag = REG_EXTENDED;

	if ( regcomp((regex_t*)_pRE, (char*)_expr, flag) != 0 )
	{
	    throw Exception(EXLOC, "regcomp error"); 
	}
    }
    else
    {
	Chain msg = Chain("Invalid regular expression <") + _expr + Chain(">"); 
	throw Exception(EXLOC, msg); 
    }
    _isPrepared = true;    
}

bool Matcher::match(const Chain& str)
{
    if ( _isPrepared )
    {
	if ( (char*)str != 0  )
	    if (regexec((regex_t*)_pRE, (char*)str, 0, NULL, 0) == 0)
		return true;
	return false;
    }
    throw Exception(EXLOC, "Matcher not prepared"); 
}
