///////////////////////////////////////////////////////////////////////////////
//                                                         
// Replacer.cc
// -----------
// String replacement via regular expression handling
//                                               
// Design and Implementation by Bjoern Lemke               
//                                                         
// (C)opyright 2000-2025 Bjoern Lemke
//
// IMPLEMENTATION MODULE
//
// Class: Matcher
// 
// Description: String replacement  utility class
//
// Status: CLEAN
//
///////////////////////////////////////////////////////////////////////////////

// POSIX INCLUDES
#include <stdlib.h>
#include <string.h>

// LFC INCLUDES
#include "Exception.h"
#include "Chain.h"
#include "Replacer.h"

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <regex.h>

#define REPSIGN '$'

Replacer::Replacer(const Chain& expr, const Chain& replace)
{    
    _expr = expr;
    _replace = replace;
    _pRE = 0;
    _m = 0;
}

Replacer::~Replacer()
{
    if ( _pRE )
    {
	regfree((regex_t*)_pRE);
	free(_pRE);
    }
    if ( _m )
	free(_m);
}

void Replacer::set(const Chain& expr, const Chain& replace)
{
    _expr = expr;
    _replace = replace;

    if (  _pRE )
    {
	regfree((regex_t*)_pRE);
	free(_pRE);
	_pRE = 0;
    }
    if ( _m )
    {
	free(_m);
	_m = 0;
    }
}

const Chain& Replacer::getExpr() const
{
    return _expr;
}

const Chain& Replacer::getReplace() const
{
    return _replace;
}

void Replacer::prepare()
{
    if (  _pRE )
    {
	regfree((regex_t*)_pRE);
	free(_pRE);
	_pRE = 0;
    }
    
    if ( _m )
    {
	free(_m);
	_m = 0;
    }

    if ( (char*)_expr != 0 )
    {
	_pRE = (regex_t*)malloc(sizeof(regex_t));

	int err;
	if ( ( err = regcomp((regex_t*)_pRE, (char*)_expr, REG_EXTENDED) ) == 0 )
	{	 
	    _nmatch = _pRE->re_nsub;
	    
	    _m = (regmatch_t*)malloc((_nmatch + 1)*sizeof(regmatch_t));
			    
	    char *p;
	    
            // count back references in replace
	    _br = 0;
	    p = (char*)_replace;
	    while( *p )
	    {		
		if (*p == REPSIGN )
		{

		    p++;		    
		    unsigned c = (unsigned)*p - 48;

		    if ( c > 0 && c < 10 )
		    {			
			_br++;			
			if ( c > _nmatch )
			{
			    Chain msg = Chain("Invalid replacement reference");
			    throw Exception(EXLOC, msg); 
			}
		    }		    
		}
		else
		    p++;		
	    }
	}
	else
	{
	    Chain msg = Chain("Regcomp error " + Chain(err));
	    throw Exception(EXLOC, msg); 
	}	
    }
    else
    {
	Chain msg = Chain("Invalid regular expression <") + _expr + Chain(">"); 
	throw Exception(EXLOC, msg); 
    }
}

unsigned long Replacer::replace(const Chain& src, Chain& target)
{
    if ( _pRE )
    {
	unsigned long replacements = 0;

	unsigned long start=1;

	// cout << "Starting regex for " << src << " with nmatch = " << _nmatch << endl;

	bool goOn = true;

	
	while( *((char*)src + start - 1) && goOn)
	{	    
	    int  err = regexec(_pRE, (char*)src + start - 1, _nmatch + 1, _m, REG_NOTBOL);
	    // cout << "Err = " << err << "REG_NOMATCH = " << REG_NOMATCH << endl;
	    
	   
	    if ( err == 0 )
	    {
		/*
		cout << "Starting regex for " << (char*)src + start - 1 << " Start = " << start << endl;
		cout << "Setting target from " << start << " to " << _m[0].rm_so << endl;
		cout << "End offset = " << _m[0].rm_eo << endl;
		*/
		
		// more matches ?
		if ( _m[0].rm_eo == 0 )
		{
		    // cout << "Detecing end " << endl;
		    goOn = false;
		}
		else
		{
		
		    // if available, add prefix 
		    if ( _m[0].rm_so > 0 )
			target += src.subChain(start, start + _m[0].rm_so - 1);
		
		    
		    // cout << "Target 0 = " << target << endl;
		    
		    char* pStart = (char*)_replace;
		    char* p = pStart;
		    
		    unsigned long rplstart = 0;
		    unsigned long rplend = p - pStart + 1;
		    
		    while ( *p )
		    {
			if ( *p == REPSIGN )
			{		    		    
			    p++;
			    unsigned c = *p - 48 ;
			    
			    rplend = p - pStart - 1;
			    
			    if ( c > 0 && c < 10 )
			    {			
				// cout << "Target 1 = " << target << " from " << rplstart << " to " << rplend <<  endl;
				
				// if available, add replace pattern prefix
				if ( rplstart > 0 && rplend >= rplstart )
				    target += _replace.subChain(rplstart, rplend);
				
				// cout << "Target 1a = " << target << " from  = " << _m[c].rm_so << " to " << _m[c].rm_eo  <<  endl;
				
				// if available, add replacement string
				if ( _m[c].rm_eo > _m[c].rm_so )
				    target += src.subChain(start+_m[c].rm_so, start+_m[c].rm_eo-1);
				
				// cout << "Target 2 = " << target << endl;
				
				p++; 
				
				rplstart=rplend+3;
				rplend=rplstart;
			    }
			    else
			    {					     
				p++;
				
				// cout << "Target X = " << target << " " << rplstart << " " << rplend <<  endl;
				if ( rplstart > 0 && rplend >= rplstart )
				    target += _replace.subChain(rplstart, rplend);
				
				// we take the $ sign with next chunk
				rplstart=rplend+2;
				rplend=rplstart;
			    }		    
			}
			else 
			{			    
			    if ( rplstart == 0 )
			    {
				rplstart = p - pStart + 1;
				rplend = p - pStart  + 1 ;
			    }
			    
			    p++;		    
			}
		    }
		    		    
		    // cout << "Target 3 = " << target << endl;		    
		    // cout << "rplend = " << rplend << " replace len = " << _replace.length() << endl;
		    
		    // if available, add replacement postfix
		    if ( rplend < _replace.length() )
		    {
			target += _replace.subChain(rplend, _replace.length());
		    }
		    
		    // cout << "Target 4 = " << target << endl;		    
		    
		    start += _m[0].rm_eo;
		    
		    replacements++;
		    
		}
	    }
	    else if ( err == REG_NOMATCH )
	    {
		goOn = false;		
	    }
	    else // if ( err > 1)
	    {
		throw Exception(EXLOC, Chain("regex error = ") + Chain(err)); 
	    }
	}

        // cout << "Adding tail from " << start << " to " << src.length() << endl;

	// if available, add source suffix
	if ( start > 0 && src.length() >= start )
	    target += src.subChain(start, src.length());

	// cout << "Target 5 = " << target << endl;
	
        return replacements;
    }
    throw Exception(EXLOC, "Replacer not prepared"); 
}
