#ifndef _PAGER_H_INCLUDED_ 
#define _PAGER_H_INCLUDED_ 
/////////////////////////////////////////////////////////////////////////////// 
// 
// Pager.h 
// -------
// Curses based pager interface definition ( second generation ) 
// 
// Design and Implementation by Bjoern Lemke 
// 
// (C)opyright 2025 by Bjoern Lemke 
// 
// INTERFACE MODULE 
// 
// Class: CegoAdmMon 
// 
// Description: Description: Cursor based monitor 
// 
// Status: CLEAN 
// 
/////////////////////////////////////////////////////////////////////////////// 

#include <ctype.h> 
#include <string.h> 

#ifndef LFCNOCURSES 
#include <ncurses.h> 
#endif 

#include "Chain.h" 
#include "ListT.h" 
#include "Datetime.h" 
#include "SigHandler.h"
#include "Matcher.h" 

#define MAXCMDROW 30
#define MAXCMDCOL 300

class Pager : public SigHandler { 

public: 

   
    Pager(const Chain& title);
    ~Pager(); 
    
    virtual bool execute(const Chain& cmd, Chain& msg) = 0;    

    virtual bool hasData() = 0;
    virtual ListT<Chain> getSchema() = 0;
    virtual ListT< ListT<Chain> > getData() = 0;

    void setHistSep(const Chain& histSep);
    void readHistory(const Chain& historyFile);
    void writeHistory(const Chain& historyFile);

    void manage();
    
private: 

    void setPager( const ListT<Chain>& schema, const ListT< ListT<Chain> >& outData );
    void managePager();
    
    bool nextRow();
    bool prevRow();

    void firstRow();
    void lastRow();
    
    void writeHeader(); 
    void writePage();
    void writeFooter();
    void writeDebug() ;
    void refreshRow(int dataRow, int pageRow);
    void getAttrSpec(int colPos, const Chain& s, Chain& attr, int& len);
    void sigCatch(int sig);

    void formatData(const ListT< ListT<Chain> > & outData); 

    void addInput(int c);
    bool isCmdInput(int c);
    Chain getCommand();
    void setCommand(const Chain& cmd);
    
    void showHelp(int helpMode);
    void showMessage();
    bool showSearch();
    bool nextSearchResult(bool isFirst);
    bool matchRow(int row);

    int addHist(const Chain& cmd);

    void initCmd();
    
    Matcher _matcher;
    
    ListT<Chain> _schema;
    ListT<int> _outLen;
    ListT<int> _rowHeight;
    ListT< ListT<Chain> > _outData;

    ListT<Chain> _cmdHist;
    Chain _histSep;
    
    Chain _title;
    Chain _debug;

    Chain _mode;
    Chain _status;
    Chain _msg;
    Chain _searchPattern;
    
    int _rows;
    int _cols;

    int _curRefresh;

    int _footerSize; 
    
    int _pdi; // previous data idx
    int _pci; // previous cursor idx
    int _ndi; // next data idx
    int _nci; // next cursor
    
    int _rowIdx;
    int _rowSelected;
    int _vStart;
    //int _offset;

    int _cntrl;
    int _curCol;
    int _curRow;
    int _curOffset;

    int _histPos;

    char _cmdInput[MAXCMDROW][MAXCMDCOL];

    bool _terminate;
    bool _activeSearch;
    
    WINDOW* _helpwin;
}; 

#endif
