///////////////////////////////////////////////////////////////////////////////
//                                                         
// NetHandler.cc
// -------------
// Net handler for sending and receiving messages
//                                               
// Design and Implementation by Bjoern Lemke               
//                                                         
// (C)opyright 2000-2013 Bjoern Lemke                        
//
// IMPLEMENTATION MODULE
//
// Class: NetHandler
// 
// Description: Net handle for sending and receiving messages
//
// Status: CLEAN
//
///////////////////////////////////////////////////////////////////////////////


#ifndef _REENTRANT
#define _REENTRANT    /* basic 3-lines for threads */
#endif

#include <string.h>

#ifdef HAVE_MINGW
#include <winsock2.h>
#include <windows.h>
#define CLOSESOCKET ::closesocket
#else
#include <unistd.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <netdb.h>
#define TIMEVAL struct timeval
#define CLOSESOCKET ::close
#endif

#include <errno.h>

#include "Exception.h"
#include "Net.h"
#include "NetHandler.h"

// on FreeBSD based systems, a SENDLEN value of 1024 byte
// lead to poor performance for large messages
// So for modern network stacks a sendlen of 8192 might be useful

// #define SENDLEN 8192

/*
NetHandler::NetHandler(int initMsgBufLen, int sizeInfoLen, int maxSendLen)
{
    _socket=0;
    _msgSize=0;
    _msgBufSize = initMsgBufLen;
    _sizeInfoLen = sizeInfoLen;
    _maxSendLen = maxSendLen;
    
    _concatPos = 0;

    _msgBuf = new char[_msgBufSize + _sizeInfoLen];
}
*/

NetHandler::NetHandler(int s, int initMsgBufLen, int sizeInfoLen, int maxSendLen)
{
    _socket=s;
    _msgSize=0;
    _msgBufSize = initMsgBufLen;
    _sizeInfoLen = sizeInfoLen;
    _maxSendLen = maxSendLen;
    _msgBuf = new char[_msgBufSize + _sizeInfoLen];
}

NetHandler::~NetHandler()
{
    if ( _socket )
	CLOSESOCKET(_socket);
    delete _msgBuf;
}

int NetHandler::getSocket()
{
    return _socket;
}

void NetHandler::setSource(const Chain& source)
{
    _source = source;
}

const Chain& NetHandler::getSource() const
{
    return _source;
}

bool NetHandler::waitMsg(int ms)
{
    int result;

    TIMEVAL tv;
    
    tv.tv_sec = ms / 1000 ;
    tv.tv_usec = ( ms % 1000) * 1000  ;
        
    fd_set readset;
    
    do
    {
	FD_ZERO(&readset);
	FD_SET(_socket, &readset);
	result = ::select(_socket + 1, &readset, NULL, NULL, &tv);
    }
    while (result == -1 && errno == EINTR);
    
    if (result > 0) 
    {
	if (FD_ISSET(_socket, &readset)) 
	{
	    return true;
	}
    }
    else if (result < 0) 
    {
	Chain msg = Chain("select system error : ") + Chain(strerror(errno));
	throw Exception(EXLOC, msg);	
    }
    
    return false;
}
    
void NetHandler::readMsg()
{
    
    int recvByte;
    if ( (recvByte = ::recv(_socket,_msgBuf, _msgBufSize + _sizeInfoLen , 0 )) <= 0 )
    {
	Chain msg = Chain("recv system error : ") + Chain(strerror(errno));
	throw Exception(EXLOC, msg);
    }

    int i = 0;
    while ( _msgBuf[i] != '@' && i < _sizeInfoLen)
    {
	i++;
    }
    Chain sizeInfo(_msgBuf, i);

    _msgSize = sizeInfo.asInteger();
    
    if ( _msgSize + 1 > _msgBufSize )
    {
	char* copyBuf = _msgBuf;
	_msgBufSize = _msgSize + 1; // we need plus 1 to terminate message with zero character
	_msgBuf = new char[_msgBufSize + _sizeInfoLen];
	memcpy(_msgBuf, copyBuf, recvByte );
	delete copyBuf;

    }
    while ( recvByte < _msgSize + _sizeInfoLen )
    {
	int i;
	if ( ( i = ::recv(_socket, (char*)((long long)_msgBuf + (long)recvByte), ( _msgSize + _sizeInfoLen ) - recvByte, 0) ) <= 0 )
	{
	    Chain msg = Chain("recv system error : ") + Chain(strerror(errno));
	    throw Exception(EXLOC, msg);
	}
	recvByte += i;
    }

    // terminate msg in every case
    _msgBuf[_msgSize + _sizeInfoLen ]=0;

    // char* msg = (char*)( (long long)_msgBuf + _sizeInfoLen);
    // cout << "Msg =  " << msg << endl;

    return;
}

void NetHandler::writeMsg()
{
    // first send msg size

    Chain sizeInfo(_msgSize);
   
    if ( (int)sizeInfo.length() > _sizeInfoLen )
    {
	Chain msg = Chain("Message too long : ") + sizeInfo;
	throw Exception(EXLOC, msg);
    }
    
    memcpy(_msgBuf, (char*)sizeInfo, sizeInfo.length() - 1 );
    
    int i = sizeInfo.length() - 1;
    while ( i < _sizeInfoLen)
    {
	_msgBuf[i] = '@';
	i++;
    }
    
    int sendByte = 0;
    while ( sendByte < _msgSize + _sizeInfoLen)
    {
	int i;	
	int sendSize;

	if ( _maxSendLen < _msgSize + _sizeInfoLen - sendByte )
	    sendSize =  _maxSendLen; 
	else 
	    sendSize = _msgSize + _sizeInfoLen - sendByte;

	if ( ( i = ::send(_socket, (char*)((long long)_msgBuf + (long)sendByte), sendSize,0) ) == -1 )
	{
	    Chain msg = Chain("send system error : ") + Chain(strerror(errno));	 
	    throw Exception(EXLOC, msg);
	}	
	sendByte += i;		
    }
}

bool NetHandler::recvAck()
{
    char ack;
    int ackSize;
    if ( ( ackSize = ::recv(_socket,&ack,1,0) ) <= 0 )
    {
	Chain msg = Chain("recv system error : ") + Chain(strerror(errno));
	throw Exception(EXLOC, msg);
    }
    return ( ack == 1 );
}

void NetHandler::sendAck()
{
    char ack = 1;
    int ackSize;
    
    if ( ( ackSize = ::send(_socket, &ack,1,0)) == -1)
    {
	Chain msg = Chain("send system error : ") + Chain(strerror(errno));
	throw Exception(EXLOC, msg);
    }
}

void NetHandler::sendNack()
{
    char ack = 0;
    int ackSize;
    
    if ( ( ackSize = ::send(_socket, &ack,1,0)) == -1)
    {
	Chain msg = Chain("send system error : ") + Chain(strerror(errno));
	throw Exception(EXLOC, msg);
    }
}

void NetHandler::sendChar(char c)
{   
    if ( ::send(_socket, &c,1,0) == -1 )
    {
	Chain msg = Chain("send system error : ") + Chain(strerror(errno));
	throw Exception(EXLOC, msg);
    }
}

char NetHandler::recvChar()
{    
    char c;
    if ( ::recv(_socket,&c,1,0) <= 0 )
    {
	Chain msg = Chain("recv system error : ") + Chain(strerror(errno));
	throw Exception(EXLOC, msg);
    }
    return c;
}

char* NetHandler::getMsg()
{
    return (char*)( (long long)_msgBuf + _sizeInfoLen);
}

int NetHandler::getMsgSize()
{
    return _msgSize;
}

void NetHandler::setMsg(char* msg, int size)
{
    if ( size > _msgBufSize )
    {
	delete _msgBuf;
	_msgBufSize = size;
	_msgBuf = new char[_msgBufSize + _sizeInfoLen];
	// Chain msg = Chain("Message too long : ") + Chain(size) + Chain(" bytes");
	// throw Exception(EXLOC, msg);
    }
    if ( size > 0 )
	memcpy((char*)((long long)_msgBuf + _sizeInfoLen), msg, size);

    _msgSize=size;
}

int NetHandler::concatPos()
{
    return _concatPos;
}

void NetHandler::concatReset()
{
    _concatPos = 0;
}

void NetHandler::concatAdd(const Chain& s)
{
    concatAdd((char*)s, s.length()-1);
}

void NetHandler::concatAdd(char* msg, int size)
{
    if ( size + _concatPos > _msgBufSize )
    {

	int newSize = size + _concatPos;
	
	char *newBuf = new char[newSize + _sizeInfoLen];

	memcpy(newBuf, _msgBuf, _concatPos + _sizeInfoLen);

	delete _msgBuf;	

	_msgBufSize = newSize;	
	_msgBuf = newBuf;
    }
    if ( size > 0 )
    {
	memcpy((char*)((long long)_msgBuf + _concatPos + _sizeInfoLen), msg, size);
    }
    
    _msgSize = size + _concatPos;
    _concatPos += size;
}

void NetHandler::disconnect()
{
    if (_socket )
    {
	CLOSESOCKET(_socket);
	_socket=0;
    }
}

NetHandler& NetHandler::operator=(const NetHandler& nh)
{
    _socket = nh._socket;
    memcpy(_msgBuf, nh._msgBuf, _msgBufSize + _sizeInfoLen);
    _msgSize = nh._msgSize;
    _msgBufSize = nh._msgBufSize;
    _maxSendLen = nh._maxSendLen;
    _concatPos = nh._concatPos;

    return (*this);
}
    
bool NetHandler::operator==(const NetHandler& nh) const
{
    if (nh._socket == _socket) 
	return true;
    return false;
}
