///////////////////////////////////////////////////////////////////////////////
//                                                         
// Chain.cc
// ---------
// String class implementation
//                                               
// Design and Implementation by Bjoern Lemke               
//                                                         
// (C)opyright 2000-2016 Bjoern Lemke 
//
// IMPLEMENTATION MODULE
//
// Class: Chain
// 
// Description: All operations on strings
//
// Status: CLEAN
//
///////////////////////////////////////////////////////////////////////////////

// SYSTEM INCLUDES
#include <stdlib.h>
#include <ctype.h>
#include <string.h>
#include <errno.h>
#include <limits.h>
#include <wctype.h>

// BASE INCLUDES
#include "Exception.h"
#include "Chain.h"

// DEFINES
#define TMPBUFSIZE 100
#define INBUFSIZE 100

Chain::Chain()
{
     _buf = 0;
     _len = 0;
}

Chain::Chain(const char *s)
{
     _buf = 0;
     _len = 0;
     
     if ( s == 0 ) 
     {
	  return;
     }

     size_t sl = strlen(s) + 1;

     if ( sl < STATICBUFSIZE )
     {
	 _buf = _staticBuf;
     }
     else
     {
	 _buf = (char*)malloc(sl);
	 if ( _buf == 0 ) 
	 {
	     throw Exception(EXLOC, "Malloc system error"); 
	 }
     }
     strcpy(_buf, s); 
     _len=sl;
}

Chain::Chain(const char *s, int len)
{
     _buf = 0;
     _len = 0;
     
     if ( s == 0 ) 
     {
	  return;
     }

     if ( len+1 < STATICBUFSIZE )
     {
	 _buf = _staticBuf;
     }
     else
     {	 
	 _buf = (char*)malloc(len+1);
	 
	 if ( _buf == 0 ) 
	 {
	     throw Exception(EXLOC, "Malloc system error"); 
	 }
     }

     memcpy(_buf, s, len);
     _buf[len]=0;
     _len=len+1;
}

Chain::Chain(const Chain& str)
{    
     _len = 0;
     _buf = 0;

     *this = str;
}

Chain::Chain(const char c)
{
     _buf = 0;
     _len = 0;
     
     if ( c > 0 )
     {
	 _buf = _staticBuf;	 
	 _buf[0]=c;
	 _buf[1]=0;	 
	 _len=2;
     }
}

Chain::Chain(long l)
{    
    char tmpBuf[TMPBUFSIZE];
    
    _len = snprintf(tmpBuf, TMPBUFSIZE, "%ld", l);
    if ( _len > TMPBUFSIZE )
    {
	throw Exception(EXLOC, "Temp buf size exceeded"); 
    }
    
    _len++;

    if (_len > STATICBUFSIZE )
    {
	throw Exception(EXLOC, "Static buf size exceeded"); 
    }
    
    _buf = _staticBuf;
    
    strcpy(_buf, tmpBuf);	
}

Chain::Chain(long long ll)
{    
    char tmpBuf[TMPBUFSIZE];
    
    _len = snprintf(tmpBuf, TMPBUFSIZE, "%lld", ll);
    if ( _len > TMPBUFSIZE )
    {
	throw Exception(EXLOC, "Temp buf size exceeded");
    }

    _len++;

    if (_len > STATICBUFSIZE )
    {
	throw Exception(EXLOC, "Static buf size exceeded"); 
    }
    
    _buf = _staticBuf;
    
    strcpy(_buf, tmpBuf);
}

Chain::Chain(unsigned long l)
{    
    char tmpBuf[TMPBUFSIZE];
    
    _len = snprintf(tmpBuf, TMPBUFSIZE, "%lu", l);
    if ( _len > TMPBUFSIZE )
    {
	throw Exception(EXLOC, "Temp buf size exceeded"); 
    }

    _len++;

    if (_len > STATICBUFSIZE )
    {
	throw Exception(EXLOC, "Static buf size exceeded"); 
    }
    
    _buf = _staticBuf;
    
    strcpy(_buf, tmpBuf);
}

Chain::Chain(unsigned long long ull)
{    
    char tmpBuf[TMPBUFSIZE];
    
    _len = snprintf(tmpBuf, TMPBUFSIZE, "%llu", ull);
    if ( _len > TMPBUFSIZE )
    {
	throw Exception(EXLOC, "Temp buf size exceeded"); 
    }

    _len++;

    if (_len > STATICBUFSIZE )
    {
	throw Exception(EXLOC, "Static buf size exceeded"); 
    }
    
    _buf = _staticBuf;
    
    strcpy(_buf, tmpBuf);
}

Chain::Chain(unsigned ui)
{    
    char tmpBuf[TMPBUFSIZE];
    
    _len = snprintf(tmpBuf, TMPBUFSIZE, "%u", ui);
    if ( _len > TMPBUFSIZE )
    {
	throw Exception(EXLOC, "Temp buf size exceeded");
    }

    _len++;

    if (_len > STATICBUFSIZE )
    {
	throw Exception(EXLOC, "Static buf size exceeded"); 
    }
    
    _buf = _staticBuf;
    
    strcpy(_buf, tmpBuf);
}

Chain::Chain(int i)
{    
    char tmpBuf[TMPBUFSIZE];
    
    _len = snprintf(tmpBuf, TMPBUFSIZE, "%d", i);
    if ( _len > TMPBUFSIZE )
    {
	throw Exception(EXLOC, "Temp buf size exceeded");
    }

    _len++;

    if (_len > STATICBUFSIZE )
    {
	throw Exception(EXLOC, "Static buf size exceeded"); 
    }
    
    _buf = _staticBuf;
    
    strcpy(_buf, tmpBuf);
}

Chain::Chain(float f)
{    
    char tmpBuf[TMPBUFSIZE];

    int l = snprintf(tmpBuf, TMPBUFSIZE, "%f", f);
    if ( l > TMPBUFSIZE )
    {
	throw Exception(EXLOC, "Temp buf size exceeded"); 
    }

    if ( l < 0 )
    {
	throw Exception(EXLOC, "Invalid format for float value");
    }

    _len = l+1;

    if (_len > STATICBUFSIZE )
    {
	throw Exception(EXLOC, "Static buf size exceeded"); 
    }
    
    _buf = _staticBuf;
    
    strcpy(_buf, tmpBuf);
}

Chain::Chain(double d, const char* format)
{    
    char tmpBuf[TMPBUFSIZE];
    
    int l = snprintf(tmpBuf, TMPBUFSIZE, format, d);
    if ( l > TMPBUFSIZE )
    {
	throw Exception(EXLOC, "Temp buf size exceeded"); 
    }

    if ( l < 0 )
    {
	throw Exception(EXLOC, "Invalid format for double value");
    }
    _len = l+1;

    if ( _len > STATICBUFSIZE )
    {
	throw Exception(EXLOC, "Static buf size exceeded"); 
    }
    
    _buf = _staticBuf;
    
    strcpy(_buf, tmpBuf);
}

Chain::~Chain()
{
    if (_buf != 0 && _buf != _staticBuf) 
    {
	free(_buf);
    }
    _buf = 0;
    _len = 0;
}

void Chain::setData(char* s)
{
    _buf = s;
    _len=strlen(s)+1;
}

unsigned long Chain::length() const
{
     return _len;
}

unsigned long Chain::visibleLength() const
{
    unsigned long vlen=0;
    unsigned long i=0;

    wchar_t t;
    int result;
    while ( ( result = mbtowc(&t, _buf+i, MB_CUR_MAX)) > 0  && i < _len-1 )
    {	
	i+=result;
	vlen++;	
    }
    vlen++;
    return vlen;
}

Chain Chain::toUpper() const
{
     Chain s = *this;

     wchar_t wText[_len];

     if ( ::mbstowcs(wText, _buf, _len) == (size_t)-1 )
     {
	 throw Exception(EXLOC, "Cannot convert to wide character"); 
     }

     unsigned long i=0;
     wchar_t c;
     while (wText[i])
     {
	 c = wText[i];
	 wText[i] = towupper(c);
	 i++;
     }
     
     if ( ::wcstombs(s._buf, wText, _len) == (size_t)-1 )
     {
	 throw Exception(EXLOC, "Cannot convert to multi byte"); 
     }
          
     return s;
}

Chain Chain::toLower() const
{
     Chain s = *this;

     wchar_t wText[_len];

     if ( ::mbstowcs(wText, _buf, _len) == (size_t)-1 )
     {
	 throw Exception(EXLOC, "Cannot convert to wide character"); 
     }
     
     unsigned long i=0;
     wchar_t c;
     while (wText[i])
     {
	 c = wText[i];
	 wText[i] = towlower(c);
	 i++;
     }
     
     if ( ::wcstombs(s._buf, wText, _len) == (size_t)-1 )
     {
	 throw Exception(EXLOC, "Cannot convert to multi byte"); 
     }

     return s;
}

unsigned Chain::asUnsigned(bool doStrict) const
{
    if ( _buf == 0 )
	return 0;

    long l = strtol(_buf, NULL, 0);
    
    if ( doStrict )
    {
	if ( l == 0 && errno == EINVAL )
	{
	    Chain msg = Chain("Integer conversion error : ") + Chain(strerror(errno));
	    throw Exception(EXLOC, msg);	
	}	
    }

    if ( l > INT_MAX || l < 0 )
    {
	Chain msg = Chain("Integer conversion error : Out of range");
	throw Exception(EXLOC, msg);	
    }
	    
    unsigned ui = (unsigned)l;
    return ui;
}

int Chain::asInteger(bool doStrict) const
{
    if ( _buf == 0 )
	return 0;

    long l = strtol(_buf, NULL, 0);
    
    if ( doStrict )
    {
	if ( l == 0 && errno == EINVAL )
	{
	    Chain msg = Chain("Integer conversion error : ") + Chain(strerror(errno));
	    throw Exception(EXLOC, msg);	
	}	
    }

    if ( l > INT_MAX || l < INT_MIN )
    {
	Chain msg = Chain("Integer conversion error : Out of range");
	throw Exception(EXLOC, msg);	
    }
	    
    int i = (int)l;
    return i;
}

long Chain::asLong(bool doStrict) const
{
    if ( _buf == 0 )
	return 0;	

    if ( doStrict )
    {
	long l = strtol(_buf, NULL, 0);
	if ( l == 0 && errno == EINVAL )
	{
	    Chain msg = Chain("Long conversion error : ") + Chain(strerror(errno));
	    throw Exception(EXLOC, msg);	
	}
	return l;    
    }
    else
    {
	return strtol(_buf, NULL, 0);
    }
}

unsigned long Chain::asUnsignedLong(bool doStrict) const
{
    if ( _buf == 0 )
	return 0;	

    if ( doStrict )
    {
	unsigned long ul = strtoul(_buf, NULL, 0);
	if ( ul == 0 && errno == EINVAL )
	{
	    Chain msg = Chain("Unsigned long conversion error : ") + Chain(strerror(errno));
	    throw Exception(EXLOC, msg);	
	}
	return ul;	
    }
    else
    {
	return strtoul(_buf, NULL, 0);
    }
}

long long Chain::asLongLong(bool doStrict ) const
{
    if ( _buf == 0 )
	return 0;

    if ( doStrict )
    {
	long long ll = strtoll(_buf, NULL, 0);
	if ( ll == 0 && errno == EINVAL)
	{
	    Chain msg = Chain("Long long conversion error : ") + Chain(strerror(errno));
	    throw Exception(EXLOC, msg);	
	}
	return ll;	
    }
    else
    {
	return strtoll(_buf, NULL, 0);	
    }
}

unsigned long long Chain::asUnsignedLongLong(bool doStrict) const
{
    if ( _buf == 0 )
	return 0;	

    if ( doStrict )
    {
	unsigned long long ull = strtoull(_buf, NULL, 0);
	if ( ull == 0 && errno == EINVAL)
	{
	    Chain msg = Chain("Unsigned long long conversion error : ") + Chain(strerror(errno));
	    throw Exception(EXLOC, msg);	
	}
	return ull;
    }
    else
    {
	return strtoull(_buf, NULL, 0);
    }
}

float Chain::asFloat() const
{
    if ( _buf == 0 )
	return 0.0;

    float f;
    if ( sscanf((char*)(_buf), "%f", &f) == 0 )
	throw Exception(EXLOC, "Cannot convert to float");
    return f;
}

double Chain::asDouble() const
{
    if ( _buf == 0 )
	return (double)0.0;

    double d;
    if ( sscanf((char*)(_buf), "%lf", &d) == 0 )
    	throw Exception(EXLOC, "Cannot convert to double");
    return d;
}

short Chain::asShort() const
{
    if ( _buf == 0 )
	return 0;

    short s;
    if ( sscanf((char*)(_buf), "%hd", &s) == 0 )
	throw Exception(EXLOC, "Cannot convert to short");
    return s;
}

char Chain::asChar() const
{
    if ( _buf == 0 )
	return 0;

    char c;
    if ( sscanf((char*)(_buf), "%c", &c) == 0 )
    	throw Exception(EXLOC, "Cannot convert to char");
    return c;
}

bool Chain::asBool() const
{
    if ( _buf == 0 )
	return false;

    if ( Chain(_buf) == Chain("true") || Chain(_buf) == Chain("yes") || Chain(_buf) == Chain("Y") || atoi(_buf) > 0 )
	return true;
    return false;
}

bool Chain::isNum() const
{
    if ( _len > 1 )
    {
	unsigned long i=0;
	while ( i < _len-1 )
	{
	    if ( ! isdigit(_buf[i]) )
		return false;
	    i++;
	}
	return true;
    }
    return false;
}

bool Chain::isDec() const
{
     unsigned long i=0;
     bool dotRead = false;
     while ( i < _len-1 )
     {
	 if ( ! isdigit(_buf[i]) )
	 {
	     if ( _buf[i] == '.' && dotRead == false )
		 dotRead = true;
	     else
		 return false;
	 }
	 i++;
     }
     return true;
}

Chain Chain::cutTrailing(const Chain& str) const
{
    if ( _len > 1 )
    {
	unsigned long i,l,u;
	
	l=0;
	u=_len-1;
	i=0;
	while (i < str._len && l < _len)
	{
	    if (_buf[l] == str._buf[i])
	    {
		l++;
		i=0;
	    }
	    else
	    {
		i++;
	    }
	}
	
	i=0;
	while (i < str._len && u > 0)
	{
	    if (_buf[u] == str._buf[i])
	    {
		u--;
		i=0;
	    }
	    else
	    {
		i++;
	    }
	}
	if (l <= u)
	    return subChain(l+1, u+1);
	else
	    return Chain();
    }
    return *this;
}

Chain Chain::truncLeft(const Chain& str) const
{
    if ( _len > 1 )
    {
	unsigned long i, l;
	
	l=0;
	i=0;
	while (i < str._len && l < _len)
	{
	    if (_buf[l] == str._buf[i])
	    {
		l++;
		i=0;
	    }
	    else
	    {
		i++;
	    }
	}
	if ( l < _len )
	    return subChain(l+1, _len);
	else
	    return Chain("");
    }
    return *this;
}

Chain Chain::truncRight(const Chain& str) const
{
    if ( _len > 1 )
    {
	unsigned long i, u;

	u=_len-1;
	i=0;
	while (i < str._len && u >= 0)
	{	    
	    if (_buf[u] == str._buf[i])
	    {	
		u--;
		i=0;
	    }
	    else
	    {
		i++;
	    }
	}

	if ( u >= 0 )
	    return subChain(1, u+1);
	else
	    return Chain("");
    }
    return *this;
}

bool Chain::posStr(const Chain& s1, int& result, int start, int occur) const
{
    int ocount=1;
    if ( start >=0 )
    {
	for (int i=start; i<(int)_len; i++)
	{
	    if ( matchAtPos(s1, i) )
	    {
		if ( ocount == occur )
		{
		    result = i + 1;
		    return true;
		}
		else
		    ocount++;
	    }
	}
    }
    else
    {
	for (int i=_len-1; i>0; i--)
	{
	    if ( matchAtPos(s1, i) )
	    {
		if ( ocount == occur )
		{
		    result = i + 1;
		    return true;
		}
		else
		    ocount++;
	    }
	}
    }
    return false;
}

bool Chain::matchAtPos(const Chain& s1, int pos) const
{
    if ( _buf[pos] == s1[0] )
    {	    
	int j=0;
	bool match = true;
	while (match && j < (int)s1.length()-1 && pos+j < (int)_len) 
	{
	    if ( _buf[pos+j] != s1[j] )
	    {
		match = false;
	    }
	    else
	    {
		j++;
	    } 
	}
	if (match && j == (int)s1.length()-1)
	{  
	    return true;		
	}
    }

    return false;
}

bool Chain::replace(const Chain& s1, const Chain& s2, Chain& result) const
{
    for (int i=0; i<(int)_len; i++)
    {
	if ( _buf[i] == s1[0] )
	{	    
	    int j=0;
	    bool match = true;
	    while (match && j < (int)s1.length()-1 && i+j < (int)_len) 
	    {
		if ( _buf[i+j] != s1[j] )
		{
		    match = false;
		}
		else
		{
		    
		    j++;
		} 
	    }
	    if (match && j == (int)s1.length()-1)
	    {
				
		Chain a, c;
		if (i > 0)
		{
		    a = subChain(1,i);
		}
		if (i+j < (int)_len)
		{
		    c = subChain(i+1+j, _len);
		}
		
		result = a + s2 + c; 

		return true;
	    }
	}
    }
    return false;
}

int Chain::replaceAll(const Chain& s1, const Chain& s2, Chain& result) const
{
    Chain target; 
    int count=0; 
    int l=0; 
    Chain c; 
    int i=0; 
    while( i < (int)_len ) 
    { 
        if ( _buf[i] == s1[0] ) 
        { 
            int j=0; 
            bool match = true; 
            while (match && j < (int)s1.length()-1 && i+j < (int)_len) 
            { 
                if ( _buf[i+j] != s1[j] ) 
                { 
                    match = false; 
                } 
                else 
                { 
                    j++; 
                } 
            } 
            if (match && j == (int)s1.length()-1) 
            { 
                Chain a; 
                if (i > l) 
                { 
                    a = subChain(l+1,i); 
                } 
                if (i+j < (int)_len) 
                { 
                    c = subChain(i+j+1, _len); 
                } 
                target += a + s2; 
                count++; 
                l=i+j; 
                i=l; 
            }
	    else
	    {
		i++;
	    }
        } 
        else 
        { 
	    i++; 
        } 
    }
    target += c; 
    
    if ( count > 0 ) 
        result = target; 
    else 
        result = *this; 
    return count; 
}

void Chain::setChar(int i, char c)
{
    _buf[i] = c;
}

int Chain::getHashPos(int hashSize) const
{    
    int val=0;
    for ( int i=0; i<(int)_len; i++)
    {
	val += (int)_buf[i] ;
    }
    return ( val * ( hashSize / 100 + 1))  % hashSize;
}

Chain Chain::subChain(int l, int u) const
{
    if (l >= 0 && u <= (int)_len && l <= u)
    {
	char *tmpBuf = (char*)malloc(u-l+2);
	
	if ( _buf == 0 ) 
	{
	    throw Exception(EXLOC, "Malloc system error"); 
	}
	else
	{
	    memcpy(tmpBuf,(char*)(this->_buf+l-1), u-l+1);
	    tmpBuf[u-l+1]='\0';
	    Chain tmpChain(tmpBuf);
	    free(tmpBuf);
	    return tmpChain;
	}
    }
    else
    {
	throw Exception(EXLOC, "String position exceeded"); 
    }
}

char Chain::operator [] (int i) const
{
    if (i < (int)_len)
    {
	return _buf[i];
    }
    else
    {
	throw Exception(EXLOC, "String position exceeded"); 
    }
}

Chain::operator char* () const
{
    return _buf;
}

Chain& Chain::operator=(const Chain& str)
{     

    if (str._len == 0)
    {
	if (_len > 0)
	{
	    if ( _buf != 0 && _buf != _staticBuf )
		free(_buf);
	    _buf = 0;
	    _len = 0;
	}
    }
    else if ( str._len <= _len) 
    {
	strcpy (_buf, str._buf);
	_len = str._len;
    }
    else
    {

	if (_len > 0)
	{
	    if ( _buf != 0 && _buf != _staticBuf )
		free(_buf);
	    _buf = 0;
	    _len = 0;
	}
	
	if ( str._len + 1 < STATICBUFSIZE )
	{
	    _buf = _staticBuf;
	}
	else
	{
	    _buf = (char*)malloc(str._len + 1);
	    
	    if ( _buf == 0 ) 
	    {
		throw Exception(EXLOC, "Malloc system error"); 
	    }
	}

	strcpy(_buf, str._buf);
	_len=str._len;
    }
    return (*this);
}

Chain& Chain::operator += (const Chain& str)
{
    if ( str._len == 0 )
    {
	// nothing to do
    }
    else if ( _len == 0 ) 
    {
	*this = str; // just copy
    }
    else
    {

	if ( _len + str._len - 1 < STATICBUFSIZE )
	{
	    strcpy((char*)(_buf + _len - 1), str._buf);
	    _len = _len + str._len -1;
	}
	else
	{
	    char *tmpBuf = (char*)malloc(_len + str._len -1);
	    if ( tmpBuf == 0 )
	    {
		throw Exception(EXLOC, "Malloc system error"); 
	    }
	    
	    strcpy(tmpBuf, _buf);

	    if ( _buf != _staticBuf )
		free(_buf);

	    _buf = tmpBuf;
	    strcpy((char*)(_buf + _len - 1), str._buf);
	    _len = _len + str._len -1;
	
	}	
    }
    
    return (*this);
}
    
bool Chain::operator==(const Chain& str) const
{
    
    if (str._buf && _buf) 
    {
	if (!strcmp(str._buf, _buf))
	{
	    return(true);
	}
	return (false);
    }
    return ( str._buf == _buf );
}
    
bool Chain::operator!=(const Chain& str) const
{
    if (str._buf && _buf) 
    {
	if (strcmp(str._buf, _buf))
	{
	    
	    return(true);
	}
	return (false);
    }
    return ( str._buf != _buf );
}

bool Chain::operator < (const Chain& str) const
{
    if (str._buf && _buf) 
    {
	if (strcmp(str._buf, _buf) > 0)
	{
	    return(true);
	}
	return (false);
    }
    return ( str._buf != _buf );
}

bool Chain::operator > (const Chain& str) const
{
    if (str._buf && _buf) 
    {
	if (strcmp(str._buf, _buf) < 0 )
	{
	    
	    return(true);
	}
	return (false);
    }
    return ( str._buf != _buf );
}

bool Chain::operator <= (const Chain& str) const
{
    if (str._buf && _buf) 
    {
	if (strcmp(str._buf, _buf) >= 0)
	{
	    return(true);
	}
	return (false);
    }
    return ( str._buf != _buf );
}

bool Chain::operator >= (const Chain& str) const
{
    if (str._buf && _buf) 
    {
	if (strcmp(str._buf, _buf) <= 0 )
	{	    
	    return(true);
	}
	return (false);
    }
    return ( str._buf != _buf );
}
    
Chain operator+( const Chain& str1, const Chain& str2)
{
    Chain str3;
    str3 = str1;
    str3 += str2;
    return (str3);
}

ostream& operator << (ostream& s, const  Chain& str)
{
    if ( str._buf != 0)
	s << str._buf;
    return s;
}

istream& operator >> (istream& s,  Chain& str)
{
    char buf[INBUFSIZE];      
    cin.getline(buf, INBUFSIZE);    
    str = Chain(buf);
    return s;
}
