///////////////////////////////////////////////////////////////////////////////
//                                                         
// Net.cc
// ------
// Base network driver
//                                               
// Design and Implementation by Bjoern Lemke               
//                                                         
// (C)opyright 2000-2016 Bjoern Lemke 
//
// IMPLEMENTATION MODULE
//
// Class: Net
// 
// Description: All operations on network resources
//
// Status: CLEAN
//
///////////////////////////////////////////////////////////////////////////////

#ifndef _REENTRANT
#define _REENTRANT    /* basic 3-lines for threads */
#endif

#include <string.h>

#ifdef HAVE_MINGW
#include <winsock2.h>
#include <windows.h>
#include <ws2tcpip.h>
#define CLOSESOCKET ::closesocket
#else
#include <unistd.h>
#include <fcntl.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <netdb.h>
#define TIMEVAL struct timeval
#define CLOSESOCKET ::close
#endif
#include <errno.h>

#include "Exception.h"
#include "Net.h"

Chain getSourceInfo(struct sockaddr *s);

Net::Net(int initMsgBufLen, int sizeBufLen, int maxSendLen)
{

#ifdef HAVE_MINGW

    WSADATA wsa;
    long rc;
    rc = WSAStartup(MAKEWORD(2,0), &wsa);

#endif

    _initMsgBufLen = initMsgBufLen;
    _sizeBufLen = sizeBufLen;
    _maxSendLen = maxSendLen;
    
    _csock=0;
}

Net::~Net()
{
    if ( _csock )
    {
	CLOSESOCKET(_csock);
    }
}

NetHandler* Net::connect(const Chain& hostname, const Chain& service)
{

#ifdef HAVE_MINGW

    if ( service.isNum() == false )
    {
	Chain msg = Chain("Service must be numeric value"); 
    	throw Exception(EXLOC, msg);
    }

    /* mingw does not seem to support getaddrinfo and friends
       in an appropriate way, so we have to use old style */

    struct sockaddr_in sa;
    struct hostent     *he;
    
    if ( ( he = gethostbyname((char*)hostname) ) == NULL)
    {
	Chain msg = Chain("Cannot resolve hostname ") + hostname;
	throw Exception(EXLOC, msg);
    }    
    int sock;
    if ( ( sock = ::socket(PF_INET, SOCK_STREAM, IPPROTO_TCP) ) < 0)
    {
	Chain msg = Chain("socket system error : ") + Chain(strerror(errno)); 
    	throw Exception(EXLOC, msg);
    }

    memcpy(&sa.sin_addr, he->h_addr_list[0], he->h_length);
    sa.sin_family = AF_INET;
    sa.sin_port = htons(service.asInteger());

    if ( ::connect(sock, (struct sockaddr *)&sa, sizeof(sa)) < 0 )
    {
	Chain msg = Chain("connect system error : ") + Chain(strerror(errno)); 
    	throw Exception(EXLOC, msg);
    }

#else

    struct addrinfo hints, *res0, *res;
    int err;
    int sock;
    
    memset(&hints, 0, sizeof(hints));
    hints.ai_socktype = SOCK_STREAM;

    hints.ai_family = PF_UNSPEC;


    if ( ( err = getaddrinfo((char*)hostname, (char*)service, &hints, &res0) ) != 0 ) 
    {
	Chain msg = Chain("Cannot adr info for ") + hostname;
	throw Exception(EXLOC, msg);
    }

    for (res=res0; res!=NULL; res=res->ai_next) 
    {
	sock = ::socket(res->ai_family, res->ai_socktype, res->ai_protocol);
	if (sock < 0) 
	{
	    continue;
	}
	
	if ( ::connect(sock, res->ai_addr, res->ai_addrlen) != 0) 
	{
	    CLOSESOCKET(sock);
	    continue;
	}
	
	break;
    }

    freeaddrinfo(res0);

    if (res == NULL) 
    {
	Chain msg = Chain("Cannot connect to ") + hostname;
	throw Exception(EXLOC, msg);
    }
    
#endif


    NetHandler* pNH = new NetHandler(sock, _initMsgBufLen, _sizeBufLen, _maxSendLen);
    return pNH;
}

NetHandler* Net::connect(const Chain& hostname, const Chain& service, int timeout)
{

#ifdef HAVE_MINGW

    if ( service.isNum() == false )
    {
	Chain msg = Chain("Service must be numeric value"); 
    	throw Exception(EXLOC, msg);
    }

    struct sockaddr_in sa;
    struct hostent     *he;
    
    if ( ( he = gethostbyname((char*)hostname) ) == NULL )
    {
	Chain msg = Chain("Cannot resolve hostname ") + hostname;
	throw Exception(EXLOC, msg);
    }    
    int sock;
    if ( ( sock = ::socket(PF_INET, SOCK_STREAM, IPPROTO_TCP) ) < 0 )
    {
	Chain msg = Chain("socket system error : ") + Chain(strerror(errno)); 
    	throw Exception(EXLOC, msg);
    }

    memcpy(&sa.sin_addr, he->h_addr_list[0], he->h_length);
    sa.sin_family = AF_INET;
    sa.sin_port = htons(service.asInteger());
    
    int res;
    u_long opt = 0;    
    res = ioctlsocket(sock, FIONBIO, &opt);
    if (res != NO_ERROR)
    {
	Chain msg = Chain("ioctlsocket system error : ") + Chain(strerror(errno));
	throw Exception(EXLOC, msg);
    }
    
    if ( ::connect(sock, (struct sockaddr *)&sa, sizeof(sa)) < 0 )
    {
	if (errno == WSAEINPROGRESS) 
	{
	   fd_set wait_set;
	   
	   FD_ZERO(&wait_set);
	   FD_SET(sock, &wait_set);
	   
	   TIMEVAL tv;
	   tv.tv_sec = timeout;
	   tv.tv_usec = 0;
	   
	   if ( ::select(sock + 1, NULL, &wait_set, NULL, &tv) < 0 )
	   {
	       Chain msg = Chain("select system error : ") + Chain(strerror(errno)); 
	       throw Exception(EXLOC, msg);
	   }
       }
       else
       {
	   Chain msg = Chain("connect system error : ") + Chain(strerror(errno)); 
	   throw Exception(EXLOC, msg);	    
       }
   }

   opt=1;
   res = ioctlsocket(sock, FIONBIO, &opt);
   if (res != NO_ERROR)
   {
       Chain msg = Chain("ioctlsocket system error : ") + Chain(strerror(errno));
       throw Exception(EXLOC, msg);
   }
   
#else

    struct addrinfo hints, *res0, *res;
    int err;
    int sock;
    
    memset(&hints, 0, sizeof(hints));
    hints.ai_socktype = SOCK_STREAM;

    hints.ai_family = PF_UNSPEC;

    if ( ( err = getaddrinfo((char*)hostname, (char*)service, &hints, &res0) ) != 0 ) 
    {
	Chain msg = Chain("Cannot get adr info for ") + hostname;
	throw Exception(EXLOC, msg);
    }

    for (res=res0; res!=NULL; res=res->ai_next) 
    {
	sock = ::socket(res->ai_family, res->ai_socktype, res->ai_protocol);
	if (sock < 0) 
	{
	    continue;
	}

	int opt;
	
	if ((opt = fcntl(sock, F_GETFL, NULL)) < 0)
	{
	    Chain msg = Chain("fcntl system error : ") + Chain(strerror(errno)); 
	    throw Exception(EXLOC, msg);	
	}
	
	if (fcntl(sock, F_SETFL, opt | O_NONBLOCK) < 0)
	{
	    Chain msg = Chain("fcntl system error : ") + Chain(strerror(errno)); 
	    throw Exception(EXLOC, msg);	
	}
	
	if ( ::connect(sock, res->ai_addr, res->ai_addrlen) != 0) 
	{

	    if (errno == EINPROGRESS) 
	    {
		fd_set wait_set;
		
		FD_ZERO(&wait_set);
		FD_SET(sock, &wait_set);
		
		// struct timeval tv;
		TIMEVAL tv;
		tv.tv_sec = timeout;
		tv.tv_usec = 0;
		
		if ( ::select(sock + 1, NULL, &wait_set, NULL, &tv) < 0 )
		{
		    Chain msg = Chain("select system error : ") + Chain(strerror(errno)); 
		    throw Exception(EXLOC, msg);
		}
		
		if (fcntl(sock, F_SETFL, opt) < 0)
		{
		    Chain msg = Chain("fcntl system error : ") + Chain(strerror(errno)); 
		    throw Exception(EXLOC, msg);	
		}
	    }
	    else
	    {
		CLOSESOCKET(sock);
		continue;
	    }
	}
	   
	break;
    }

    freeaddrinfo(res0);

    if (res == NULL) 
    {
	Chain msg = Chain("Cannot connect to ") + hostname;
	throw Exception(EXLOC, msg);
    }

#endif
    
    NetHandler* pNH = new NetHandler(sock, _initMsgBufLen, _sizeBufLen, _maxSendLen);
    return pNH;
}


///////////////////////////////////////////////////////////
// Serves connections on given hostname and service      //
// Protocol and port name is figured out via getaddrinfo //
///////////////////////////////////////////////////////////

void Net::serve(const Chain& hostname, const Chain& service)
{

#ifdef HAVE_MINGW

    struct sockaddr_in sa;

    if ( ( _csock = ::socket(AF_INET,SOCK_STREAM,0) ) == 0 )
    	throw Exception(EXLOC, "socket system error");


    int optVal = 1;
    if ( setsockopt(_csock, SOL_SOCKET, SO_REUSEADDR, (char*)&optVal, sizeof(int)) )
    {
	Chain msg = Chain("setsockopt system error : ") + Chain(strerror(errno));
	throw Exception(EXLOC, msg);
    }
    
    sa.sin_family = AF_INET;
    sa.sin_addr.s_addr = INADDR_ANY;
    sa.sin_port = htons(service.asInteger());

    if ( ::bind(_csock,(struct sockaddr *)&sa, sizeof(sa)) < 0 )
    {
	Chain msg = Chain("bind system error on port ") + service + Chain(" : ") + Chain(strerror(errno));
    	throw Exception(EXLOC, msg);
    }

    if ( ::listen(_csock,3) < 0 )
    {
	Chain msg = Chain("listen system error : ") + Chain(strerror(errno)); 
    	throw Exception(EXLOC, msg);
    }

#else

    struct addrinfo hints, *res;
    int err;

    memset(&hints, 0, sizeof(hints));
    hints.ai_socktype = SOCK_STREAM;

    hints.ai_family = PF_UNSPEC;

    if ( ( err = getaddrinfo((char*)hostname, (char*)service, &hints, &res) ) != 0 ) 
    {
	Chain msg = Chain("Cannot get adr info for ") + hostname + Chain("/") + service;
	throw Exception(EXLOC, msg);
    }

    try
    {
    
	// struct in6_addr in6addr_any = IN6ADDR_ANY_INIT;

	if ( (_csock=socket(res->ai_family,res->ai_socktype,res->ai_protocol)) == 0)
	    throw Exception(EXLOC, "socket system error");
	
	int optVal = 1;
	if ( setsockopt(_csock, SOL_SOCKET, SO_REUSEADDR, &optVal, sizeof(int)) )
	{
	    Chain msg = Chain("setsockopt system error : ") + Chain(strerror(errno));
	    throw Exception(EXLOC, msg);
	}
	
	/* struct sockaddr_in6 sin6;
	   
	   bzero(&sin6, sizeof(struct sockaddr_in6));
	   sin6.sin6_family = AF_INET6;
	   sin6.sin6_port = htons(port);
	   sin6.sin6_addr = in6addr_any;
	*/
	
	if ( ::bind(_csock, res->ai_addr, res->ai_addrlen) < 0 )
	{
	    Chain msg = Chain("bind system error on service ") + service + Chain(" : ") + Chain(strerror(errno));
	    throw Exception(EXLOC, msg);
	}
	
	if ( ::listen(_csock,3) < 0 )
	{
	    Chain msg = Chain("listen system error : ") + Chain(strerror(errno)); 
	    throw Exception(EXLOC, msg);
	}
    }
    catch ( Exception e )
    {
	freeaddrinfo(res);
	throw e;
    }
#endif
}


//////////////////////////////////////
// Serves v4 connetions on any host //
//////////////////////////////////////

void Net::serve(int port)
{
    struct sockaddr_in sa;

    if ( ( _csock = ::socket(AF_INET,SOCK_STREAM,0) ) == 0 )
    	throw Exception(EXLOC, "socket system error");

    int optVal = 1;

#ifdef HAVE_MINGW
    if ( setsockopt(_csock, SOL_SOCKET, SO_REUSEADDR, (char*)&optVal, sizeof(int)) )
#else
    if ( setsockopt(_csock, SOL_SOCKET, SO_REUSEADDR, &optVal, sizeof(int)) )
#endif
    {
	Chain msg = Chain("setsockopt system error : ") + Chain(strerror(errno));
	throw Exception(EXLOC, msg);
    }
    
    sa.sin_family = AF_INET;
    sa.sin_addr.s_addr = INADDR_ANY;
    sa.sin_port = htons(port);

    if ( ::bind(_csock,(struct sockaddr *)&sa, sizeof(sa) ) < 0 )
    {
	Chain msg = Chain("bind system error on port ") + Chain(port) + Chain(" : ") + Chain(strerror(errno));
    	throw Exception(EXLOC, msg);
    }

    if ( ::listen(_csock,3) < 0 )
    {
	Chain msg = Chain("listen system error : ") + Chain(strerror(errno)); 
    	throw Exception(EXLOC, msg);
    }
}

/////////////////////////////////////////////
// Serves v4 and v6 connetions on any host //
/////////////////////////////////////////////

void Net::serve6(int port)
{
    struct sockaddr_in6 sa;

    if ( ( _csock = ::socket(AF_INET6,SOCK_STREAM,0) ) == 0 )
    	throw Exception(EXLOC, "socket system error");

    int optVal = 1;

#ifdef HAVE_MINGW
    if ( setsockopt(_csock, SOL_SOCKET, SO_REUSEADDR, (char*)&optVal, sizeof(int)) )
#else
    if ( setsockopt(_csock, SOL_SOCKET, SO_REUSEADDR, &optVal, sizeof(int)) )
#endif
    {
	Chain msg = Chain("setsockopt system error : ") + Chain(strerror(errno));
	throw Exception(EXLOC, msg);
    }

    // we force double stack here
    int v6Only = 0;
    if ( setsockopt(_csock, IPPROTO_IPV6, IPV6_V6ONLY, (void *)&v6Only, sizeof(v6Only)) )
    {
	Chain msg = Chain("setsockopt system error : ") + Chain(strerror(errno));
	throw Exception(EXLOC, msg);
    }
    
    sa.sin6_family = AF_INET6;
    sa.sin6_addr = in6addr_any;
    sa.sin6_port = htons(port);

    if ( ::bind(_csock,(struct sockaddr *)&sa, sizeof(sa) ) < 0 )
    {
	Chain msg = Chain("bind system error on port ") + Chain(port) + Chain(" : ") + Chain(strerror(errno));
    	throw Exception(EXLOC, msg);
    }

    if ( ::listen(_csock,3) < 0 )
    {
	Chain msg = Chain("listen system error : ") + Chain(strerror(errno)); 
    	throw Exception(EXLOC, msg);
    }
}


/* 
   with nextRequest, incoming requests are accepted on a well known port. Further
   incoming messages from the partner are handled outside the method ( in a dedicated thread ) 
*/

NetHandler* Net::nextRequest(int timeout)
{
   
    int s;
    // struct timeval t;
    TIMEVAL t;
    t.tv_sec = timeout / 1000000;
    t.tv_usec = timeout % 1000000;

    fd_set fdSet;
    
    FD_ZERO(&fdSet);
    FD_SET(_csock, &fdSet);
    
    if ( ( s = ::select(_csock + 1, &fdSet, NULL, NULL, &t) ) < 0 )
    {
	Chain msg = Chain("select system error : ") + Chain(strerror(errno)); 
	throw Exception(EXLOC, msg);
    }

    int nsocket = 0;
    
    if ( s > 0 )
    {
	if ( FD_ISSET(_csock, &fdSet) )
	{
	    
	    // cout << "New connecction req detected .." << endl;
	    FD_CLR(_csock, &fdSet);
	    
	    struct sockaddr_in sa;

#ifdef HAVE_MINGW
	    int sa_size=sizeof(sa);
#else
	    socklen_t sa_size=sizeof(sa);
#endif
	    
	    // check for new connections ( non-blocking )
	    if ( ( nsocket = ::accept(_csock, (struct sockaddr *)&sa, &sa_size) ) < 0 )
	    {
#ifdef HAVE_MINGW
		if ( errno != 0 )
		{
		    Chain msg = Chain("accept system error: ") + Chain(strerror(errno));
		    throw Exception(EXLOC, msg);
		}
#else
		if ( errno != EWOULDBLOCK )
		{
		    Chain msg = Chain("accept system error: ") + Chain(strerror(errno));
		    throw Exception(EXLOC, msg);
		}
#endif
	    }
	    if ( nsocket > 0 )
	    {
		// cout << "Accepting new connecction ..." << nsocket  << endl;

		NetHandler* pNetHandle = new NetHandler(nsocket, _initMsgBufLen, _sizeBufLen, _maxSendLen);

		pNetHandle->setSource(getSourceInfo( (struct sockaddr *)&sa ));
		
		try 
		{
		    pNetHandle->readMsg();
		}
		catch ( Exception e)
		{	    
		    pNetHandle->disconnect();
		    delete pNetHandle;   
		    return 0;
		}
		return pNetHandle;
	    }				
	}
    }
    return 0;
}

Chain getSourceInfo(struct sockaddr *s)
{
    struct sockaddr_in *sin = (struct sockaddr_in *)s;

    /* ntoa just works for ipv4 
    cout << "ntoa result : " << endl;
    cout << inet_ntoa (sin->sin_addr);
    cout << endl;
    */
    
    char ip[INET6_ADDRSTRLEN];
    uint16_t port;

    void *addr;
    char *ipver;
    
    if (sin->sin_family == AF_INET)	
    {
	// IPv4
	// cout << "Detected v4 address" << endl;
	struct sockaddr_in *ipv4 = (struct sockaddr_in *)sin;
	addr = &(ipv4->sin_addr);
    }
    else	
    {
	// IPv6
	// cout << "Detected v6 address" << endl;
	struct sockaddr_in6 *ipv6 = (struct sockaddr_in6 *)sin;
	addr = &(ipv6->sin6_addr);
    }
    
    return Chain( inet_ntop(sin->sin_family, addr, ip, sizeof(ip)) );

}
