# frozen_string_literal: true

RSpec.shared_examples 'backfills occurrence id from vulnerabilities' do
  let(:batch_column) { :id }
  let(:connection) { SecApplicationRecord.connection }
  let(:model) { table(batch_table, database: :sec) }
  let(:start_id) { model.minimum(batch_column) }
  let(:end_id) { model.maximum(batch_column) }

  let(:organizations) { table(:organizations) }
  let(:namespaces) { table(:namespaces) }
  let(:projects) { table(:projects) }
  let(:users) { table(:users) }
  let(:identifiers) { table(:vulnerability_identifiers, database: :sec) }
  let(:scanners) { table(:vulnerability_scanners, database: :sec) }
  let(:findings) { table(:vulnerability_occurrences, database: :sec) }
  let(:vulnerabilities) { table(:vulnerabilities, database: :sec) }

  let(:organization) do
    organizations.create!(
      name: 'test',
      path: 'test'
    )
  end

  let(:scanner) do
    scanners.create!(
      created_at: now,
      updated_at: now,
      project_id: project.id,
      external_id: 'semgrep',
      name: 'Semgrep'
    )
  end

  let(:now) { Time.zone.now }
  let(:user) { users.create!(projects_limit: 1, organization_id: organization.id) }
  let(:project) { create_project }
  let!(:vulnerability) { create_vulnerability }

  let(:migration) do
    described_class.new(
      start_id: start_id,
      end_id: end_id,
      batch_table: batch_table,
      batch_column: batch_column,
      sub_batch_size: 100,
      pause_ms: 2,
      connection: connection,
      job_arguments: []
    )
  end

  subject(:perform_migration) { migration.perform }

  it 'backfills the expected records' do
    expect { perform_migration }.to change { record.reload.vulnerability_occurrence_id }
      .from(nil).to(vulnerability.finding_id)
  end

  def create_project
    namespace = namespaces.create!(
      name: 'test',
      path: 'test',
      organization_id: organization.id
    )

    projects.create!(
      namespace_id: namespace.id,
      project_namespace_id: namespace.id,
      organization_id: organization.id
    )
  end

  def create_vulnerability
    primary_identifier = identifiers.create!(
      created_at: now,
      updated_at: now,
      project_id: project.id,
      fingerprint: '0',
      external_type: 'CWE',
      external_id: 'CWE-1',
      name: 'Injection'
    )

    finding = findings.create!(
      created_at: now,
      updated_at: now,
      uuid: SecureRandom.uuid,
      severity: 1,
      report_type: 1,
      project_id: project.id,
      scanner_id: scanner.id,
      primary_identifier_id: primary_identifier.id,
      location_fingerprint: '0',
      name: 'Test vulnerability',
      metadata_version: '1'
    )

    vulnerabilities.create!(
      project_id: project.id,
      finding_id: finding.id,
      author_id: user.id,
      created_at: now,
      updated_at: now,
      title: 'Test vulnerability',
      severity: 1,
      report_type: 1
    )
  end
end
