import { GlFilteredSearch, GlKeysetPagination, GlSorting } from '@gitlab/ui';
import Vue, { nextTick } from 'vue';
import VueApollo from 'vue-apollo';
import { shallowMountExtended } from 'helpers/vue_test_utils_helper';
import createMockApollo from 'helpers/mock_apollo_helper';
import waitForPromises from 'helpers/wait_for_promises';
import { createAlert } from '~/alert';
import CrudComponent from '~/vue_shared/components/crud_component.vue';
import PersonalAccessTokensApp from '~/personal_access_tokens/components/app.vue';
import PersonalAccessTokensTable from '~/personal_access_tokens/components/personal_access_tokens_table.vue';
import PersonalAccessTokenDrawer from '~/personal_access_tokens/components/personal_access_token_drawer.vue';
import CreatePersonalAccessTokenButton from '~/personal_access_tokens/components/create_personal_access_token_button.vue';
import PersonalAccessTokenStatistics from '~/personal_access_tokens/components/personal_access_token_statistics.vue';
import getUserPersonalAccessTokens from '~/personal_access_tokens/graphql/get_user_personal_access_tokens.query.graphql';
import { DEFAULT_SORT, PAGE_SIZE } from '~/personal_access_tokens/constants';
import { mockTokens, mockPageInfo, mockQueryResponse } from '../mock_data';

jest.mock('~/alert');

Vue.use(VueApollo);

describe('PersonalAccessTokensApp', () => {
  let wrapper;
  let mockApollo;

  const mockQueryHandler = jest.fn().mockResolvedValue(mockQueryResponse);

  const createComponent = ({ queryHandler = mockQueryHandler } = {}) => {
    mockApollo = createMockApollo([[getUserPersonalAccessTokens, queryHandler]]);

    window.gon = { current_user_id: 123 };

    wrapper = shallowMountExtended(PersonalAccessTokensApp, {
      apolloProvider: mockApollo,
      stubs: {
        CrudComponent,
      },
    });
  };

  const findFilteredSearch = () => wrapper.findComponent(GlFilteredSearch);
  const findSorting = () => wrapper.findComponent(GlSorting);
  const findTable = () => wrapper.findComponent(PersonalAccessTokensTable);
  const findPagination = () => wrapper.findComponent(GlKeysetPagination);
  const findCreateButton = () => wrapper.findComponent(CreatePersonalAccessTokenButton);
  const findDrawer = () => wrapper.findComponent(PersonalAccessTokenDrawer);
  const findStatistics = () => wrapper.findComponent(PersonalAccessTokenStatistics);

  beforeEach(() => {
    createComponent();
  });

  it('renders the filtered search component', () => {
    expect(findFilteredSearch().exists()).toBe(true);
    expect(findFilteredSearch().props('availableTokens')).toBeDefined();
  });

  it('renders the sorting component', () => {
    expect(findSorting().exists()).toBe(true);
    expect(findSorting().props()).toMatchObject({
      isAscending: DEFAULT_SORT.isAsc,
      sortBy: DEFAULT_SORT.value,
    });
  });

  it('renders the create button', () => {
    expect(findCreateButton().exists()).toBe(true);
  });

  it('renders the table component', () => {
    expect(findTable().exists()).toBe(true);
  });

  describe('GraphQL query', () => {
    it('fetches tokens on mount', async () => {
      expect(findTable().props('loading')).toBe(true);

      await waitForPromises();

      expect(mockQueryHandler).toHaveBeenCalledWith({
        id: 'gid://gitlab/User/123',
        sort: 'EXPIRES_ASC',
        state: 'ACTIVE',
        first: PAGE_SIZE,
        after: null,
        last: null,
        before: null,
      });
    });

    it('passes tokens to the table', async () => {
      await waitForPromises();

      expect(findTable().props('loading')).toBe(false);
      expect(findTable().props('tokens')).toEqual(mockTokens);
    });

    it('shows alert on error', async () => {
      const errorHandler = jest.fn().mockRejectedValue(new Error('GraphQL error'));
      createComponent({ queryHandler: errorHandler });
      await waitForPromises();

      expect(createAlert).toHaveBeenCalledWith({
        message: 'An error occurred while fetching the tokens.',
        variant: 'danger',
      });
    });
  });

  describe('filtering', () => {
    it('refetches tokens when filter is cleared', async () => {
      findFilteredSearch().vm.$emit('clear');
      await nextTick();

      expect(mockQueryHandler).toHaveBeenCalledWith({
        id: 'gid://gitlab/User/123',
        sort: 'EXPIRES_ASC',
        first: PAGE_SIZE,
        after: null,
        last: null,
        before: null,
      });
    });

    it('refetches when date field with less than operator is set', async () => {
      findFilteredSearch().vm.$emit('input', [
        { type: 'expires', value: { data: '2026-01-20', operator: '<' } },
      ]);

      findFilteredSearch().vm.$emit('submit');

      await nextTick();

      expect(mockQueryHandler).toHaveBeenCalledWith({
        id: 'gid://gitlab/User/123',
        sort: 'EXPIRES_ASC',
        first: PAGE_SIZE,
        after: null,
        last: null,
        before: null,
        expiresBefore: '2026-01-20',
      });
    });

    it('refetches when date field with greater than or equal to operator is set', async () => {
      findFilteredSearch().vm.$emit('input', [
        { type: 'expires', value: { data: '2026-01-20', operator: '≥' } },
      ]);

      findFilteredSearch().vm.$emit('submit');

      await nextTick();

      expect(mockQueryHandler).toHaveBeenCalledWith({
        id: 'gid://gitlab/User/123',
        sort: 'EXPIRES_ASC',
        first: PAGE_SIZE,
        after: null,
        last: null,
        before: null,
        expiresAfter: '2026-01-20',
      });
    });

    it('refetches when when search term is set', async () => {
      findFilteredSearch().vm.$emit('input', [
        { type: 'filtered-search-term', value: { data: 'token name' } },
      ]);

      findFilteredSearch().vm.$emit('submit');

      await nextTick();

      expect(mockQueryHandler).toHaveBeenCalledWith({
        id: 'gid://gitlab/User/123',
        sort: 'EXPIRES_ASC',
        first: PAGE_SIZE,
        after: null,
        last: null,
        before: null,
        search: 'token name',
      });
    });

    it('refetches when filter is submitted', async () => {
      findFilteredSearch().vm.$emit('input', [
        { type: 'state', value: { data: 'INACTIVE' } },
        { type: 'revoked', value: { data: true } },
      ]);

      findFilteredSearch().vm.$emit('submit');

      await nextTick();

      expect(mockQueryHandler).toHaveBeenCalledWith({
        id: 'gid://gitlab/User/123',
        sort: 'EXPIRES_ASC',
        state: 'INACTIVE',
        revoked: true,
        first: PAGE_SIZE,
        after: null,
        last: null,
        before: null,
      });
    });

    it('does not refetch if submit button is not clicked', async () => {
      await waitForPromises();
      mockQueryHandler.mockClear();

      findFilteredSearch().vm.$emit('input', [
        { type: 'state', value: { data: 'INACTIVE' } },
        { type: 'revoked', value: { data: true } },
      ]);

      await nextTick();

      expect(mockQueryHandler).not.toHaveBeenCalled();
    });
  });

  describe('sorting', () => {
    it('updates sort value when changed', async () => {
      await waitForPromises();
      await findSorting().vm.$emit('sortByChange', 'name');

      expect(mockQueryHandler).toHaveBeenCalledWith({
        id: 'gid://gitlab/User/123',
        sort: 'NAME_ASC',
        state: 'ACTIVE',
        first: PAGE_SIZE,
        after: null,
        last: null,
        before: null,
      });
    });

    it('updates sort direction when changed', async () => {
      await waitForPromises();
      await findSorting().vm.$emit('sortDirectionChange', false);

      expect(mockQueryHandler).toHaveBeenCalledWith({
        id: 'gid://gitlab/User/123',
        sort: 'EXPIRES_DESC',
        state: 'ACTIVE',
        first: PAGE_SIZE,
        after: null,
        last: null,
        before: null,
      });
    });
  });

  describe('pagination', () => {
    it('shows pagination when there are more pages', async () => {
      await waitForPromises();

      expect(findPagination().exists()).toBe(true);

      expect(findPagination().props()).toMatchObject({
        startCursor: mockPageInfo.startCursor,
        endCursor: mockPageInfo.endCursor,
        hasNextPage: mockPageInfo.hasNextPage,
        hasPreviousPage: mockPageInfo.hasPreviousPage,
      });
    });

    it('hides pagination when there are no more pages', async () => {
      const noPageInfoHandler = jest.fn().mockResolvedValue({
        data: {
          user: {
            id: 'gid://gitlab/User/123',
            personalAccessTokens: {
              nodes: mockTokens,
              pageInfo: { ...mockPageInfo, hasNextPage: false },
            },
          },
        },
      });

      createComponent({ queryHandler: noPageInfoHandler });
      await waitForPromises();

      expect(findPagination().exists()).toBe(false);
    });

    it('handles next page navigation', async () => {
      await waitForPromises();
      await findPagination().vm.$emit('next', 'cursor123');

      expect(mockQueryHandler).toHaveBeenCalledWith({
        id: 'gid://gitlab/User/123',
        sort: 'EXPIRES_ASC',
        state: 'ACTIVE',
        first: PAGE_SIZE,
        after: 'cursor123',
        last: null,
        before: null,
      });
    });

    it('handles previous page navigation', async () => {
      await waitForPromises();
      await findPagination().vm.$emit('prev', 'cursor456');

      expect(mockQueryHandler).toHaveBeenCalledWith({
        id: 'gid://gitlab/User/123',
        sort: 'EXPIRES_ASC',
        state: 'ACTIVE',
        first: null,
        after: null,
        last: PAGE_SIZE,
        before: 'cursor456',
      });
    });
  });

  describe('drawer integration', () => {
    it('opens drawer when table emits select event', async () => {
      await waitForPromises();

      findTable().vm.$emit('select', mockTokens[0]);
      await nextTick();

      expect(findDrawer().props('token')).toBe(mockTokens[0]);
    });

    it('closes drawer when drawer emits close event', async () => {
      await waitForPromises();

      findTable().vm.$emit('select', mockTokens[0]);
      await nextTick();

      findDrawer().vm.$emit('close');
      await nextTick();

      expect(findDrawer().props('token')).toBe(null);
    });
  });

  describe('statistics', () => {
    it('renders the statistics component', () => {
      expect(findStatistics().exists()).toBe(true);
    });

    it('handles statistics filter events', async () => {
      await findStatistics().vm.$emit('filter', [
        {
          type: 'state',
          value: {
            data: 'ACTIVE',
            operator: '=',
          },
        },
      ]);
      await nextTick();

      expect(mockQueryHandler).toHaveBeenCalledWith({
        id: 'gid://gitlab/User/123',
        sort: 'EXPIRES_ASC',
        state: 'ACTIVE',
        first: PAGE_SIZE,
        after: null,
        last: null,
        before: null,
      });
    });
  });
});
