import Vue from 'vue';
import { shallowMount } from '@vue/test-utils';
import VueApollo from 'vue-apollo';
import { GlLoadingIcon } from '@gitlab/ui';
import createMockApollo from 'helpers/mock_apollo_helper';
import waitForPromises from 'helpers/wait_for_promises';
import ManualVariablesApp from '~/ci/pipeline_details/manual_variables/manual_variables.vue';
import EmptyState from '~/ci/pipeline_details/manual_variables/empty_state.vue';
import VariableTable from '~/ci/pipeline_details/manual_variables/variable_table.vue';
import GetManualVariablesQuery from '~/ci/pipeline_details/manual_variables/graphql/queries/get_manual_variables.query.graphql';
import { generateVariablePairs, mockManualVariableConnection } from './mock_data';

Vue.use(VueApollo);

describe('ManualVariableApp', () => {
  let wrapper;
  const mockResolver = jest.fn();
  const createMockApolloProvider = (resolver) => {
    const requestHandlers = [[GetManualVariablesQuery, resolver]];

    return createMockApollo(requestHandlers);
  };

  const createComponent = ({ variables = [], displayPipelineVariables = false } = {}) => {
    mockResolver.mockResolvedValue(mockManualVariableConnection(variables));
    wrapper = shallowMount(ManualVariablesApp, {
      provide: {
        manualVariablesCount: variables.length,
        projectPath: 'root/ci-project',
        pipelineIid: '1',
        displayPipelineVariables,
      },
      apolloProvider: createMockApolloProvider(mockResolver),
    });
  };

  const findEmptyState = () => wrapper.findComponent(EmptyState);
  const findLoadingIcon = () => wrapper.findComponent(GlLoadingIcon);
  const findVariableTable = () => wrapper.findComponent(VariableTable);

  afterEach(() => {
    mockResolver.mockClear();
  });

  describe('when component is created', () => {
    it('renders empty state when no variables were found', () => {
      createComponent();

      expect(findEmptyState().exists()).toBe(true);
    });

    it('renders empty state when variables were found, but displayPipelineVariables is false', () => {
      createComponent({ variables: generateVariablePairs(1) });

      expect(findEmptyState().exists()).toBe(true);
    });

    it('renders loading state when variables were found and displayPipelineVariables is true', () => {
      createComponent({ variables: generateVariablePairs(1), displayPipelineVariables: true });

      expect(findEmptyState().exists()).toBe(false);
      expect(findLoadingIcon().exists()).toBe(true);
      expect(findVariableTable().exists()).toBe(false);
    });

    it('renders variable table when variables were retrieved', async () => {
      createComponent({ variables: generateVariablePairs(1), displayPipelineVariables: true });
      await waitForPromises();

      expect(findLoadingIcon().exists()).toBe(false);
      expect(findVariableTable().exists()).toBe(true);
    });
  });
});
