package disk

import (
	"context"
	"io"
	"mime"
	"net/http"
	"net/url"
	"path/filepath"
	"strconv"
	"strings"

	contentencoding "gitlab.com/feistel/go-contentencoding/encoding"

	"gitlab.com/gitlab-org/gitlab-pages/internal/vfs"
)

var (
	// Server side content encoding priority.
	supportedEncodings = contentencoding.Preference{
		contentencoding.Brotli:   1.0,
		contentencoding.Gzip:     0.5,
		contentencoding.Identity: 0.1,
	}

	compressedEncodings = map[string]string{
		contentencoding.Brotli: ".br",
		contentencoding.Gzip:   ".gz",
	}
)

func endsWithSlash(path string) bool {
	return strings.HasSuffix(path, "/")
}

func endsWithoutHTMLExtension(path string) bool {
	return !strings.HasSuffix(path, ".html")
}

func cloneURL(originalURL *url.URL) *url.URL {
	newURL := new(url.URL)

	// Copy relevant fields
	*newURL = *originalURL

	return newURL
}

// Detect file's content-type either by extension or mime-sniffing.
// Implementation is adapted from Golang's `http.serveContent()`
// See https://github.com/golang/go/blob/902fc114272978a40d2e65c2510a18e870077559/src/net/http/fs.go#L194
func (reader *Reader) detectContentType(ctx context.Context, root vfs.Root, path string) (string, error) {
	ext := filepath.Ext(path)

	if contentType := mime.TypeByExtension(ext); contentType != "" {
		return contentType, nil
	}

	if contentType := reader.detectContentTypeByExtension(ext); contentType != "" {
		return contentType, nil
	}

	return reader.detectContentTypeByReading(ctx, root, path)
}

func (reader *Reader) detectContentTypeByExtension(ext string) string {
	var (
		mimeTypes = map[string]string{
			".txt": "text/plain; charset=UTF-8",
		}
	)
	contentType := mimeTypes[ext]
	return contentType
}

func (reader *Reader) detectContentTypeByReading(ctx context.Context, root vfs.Root, path string) (string, error) {
	file, err := root.Open(ctx, path)
	if err != nil {
		return "", err
	}
	defer file.Close()

	var buf [512]byte
	n, _ := io.ReadFull(file, buf[:])
	return http.DetectContentType(buf[:n]), nil
}

func (reader *Reader) handleContentEncoding(ctx context.Context, w http.ResponseWriter, r *http.Request, root vfs.Root, fullPath string) string {
	// don't accept range requests for compressed content
	if r.Header.Get("Range") != "" {
		return fullPath
	}

	acceptHeader := r.Header.Get("Accept-Encoding")

	// don't send compressed content if there's no accept-encoding header
	if acceptHeader == "" {
		return fullPath
	}

	results, err := supportedEncodings.Negotiate(acceptHeader, contentencoding.AliasIdentity)
	if err != nil {
		return fullPath
	}

	if len(results) == 0 {
		return fullPath
	}

	for _, encoding := range results {
		if encoding == contentencoding.Identity {
			break
		}

		extension := compressedEncodings[encoding]
		path := fullPath + extension

		// Ensure the file is not a symlink
		if fi, err := root.Lstat(ctx, path); err == nil && fi.Mode().IsRegular() {
			w.Header().Set("Content-Encoding", encoding)

			// http.ServeContent doesn't set Content-Length if Content-Encoding is set
			w.Header().Set("Content-Length", strconv.FormatInt(fi.Size(), 10))

			return path
		}
	}

	return fullPath
}
