package acceptance_test

import (
	"fmt"
	"net"
	"sync"
	"time"
)

// PortManager handles port assignments and tracking
type PortManager struct {
	mu             sync.Mutex
	portLastUsed   map[string]time.Time
	cooldownPeriod time.Duration
}

var globalPortManager = NewPortManager()

// NewPortManager creates a new port manager
func NewPortManager() *PortManager {
	return &PortManager{
		portLastUsed:   make(map[string]time.Time),
		cooldownPeriod: 500 * time.Millisecond,
	}
}

// WaitForPortAvailable waits until a port is available or returns immediately if unused recently
func (pm *PortManager) WaitForPortAvailable(host string, port string) {
	addr := net.JoinHostPort(host, port)

	// First check if port is actually in use
	if !IsPortInUse(host, port) {
		// Port is free, no need to check cooldown
		pm.mu.Lock()
		pm.portLastUsed[addr] = time.Now()
		pm.mu.Unlock()
		return
	}

	// Only enforce cooldown if port is actually busy
	pm.mu.Lock()
	lastUsed, found := pm.portLastUsed[addr]
	pm.mu.Unlock()

	if found {
		elapsed := time.Since(lastUsed)
		if elapsed < pm.cooldownPeriod {
			waitTime := pm.cooldownPeriod - elapsed
			fmt.Printf("Port %s is busy and was recently used, waiting %v for cooldown\n",
				addr, waitTime.Round(time.Millisecond))
			time.Sleep(waitTime)
		}
	}

	pm.mu.Lock()
	pm.portLastUsed[addr] = time.Now()
	pm.mu.Unlock()
}

// MarkPortReleased marks a port as released
func (pm *PortManager) MarkPortReleased(host string, port string) {
	addr := net.JoinHostPort(host, port)

	pm.mu.Lock()
	defer pm.mu.Unlock()

	// Mark the actual release time
	pm.portLastUsed[addr] = time.Now()
}

// IsPortInUse checks if a port is currently in use
func IsPortInUse(host string, port string) bool {
	addr := net.JoinHostPort(host, port)
	conn, err := net.DialTimeout("tcp", addr, 100*time.Millisecond)

	if err != nil {
		return false
	}

	conn.Close()
	return true
}

// WaitForPortRelease actively waits for a port to be released
func WaitForPortRelease(host string, port string, maxWait time.Duration) {
	start := time.Now()
	addr := net.JoinHostPort(host, port)

	fmt.Printf("Waiting for port %s to be released...\n", addr)

	for time.Since(start) < maxWait {
		if !IsPortInUse(host, port) {
			fmt.Printf("Port %s is now available\n", addr)
			return
		}
		time.Sleep(500 * time.Millisecond)
	}

	fmt.Printf("Warning: Port %s is still in use after waiting %v\n",
		addr, maxWait.Round(time.Second))
}
