package zoekt_test

import (
	"fmt"
	"path/filepath"
	"testing"

	"github.com/stretchr/testify/require"
	"gitlab.com/gitlab-org/gitlab-zoekt-indexer/internal/projectpath"
	"gitlab.com/gitlab-org/gitlab-zoekt-indexer/internal/zoekt"
	"go.uber.org/goleak"
)

var (
	ExistingID    uint64 = 309
	NonExistentID uint64 = 20
	ProjectID     string = "12345"
)

func TestGetCurrentSHA(t *testing.T) {
	dir := filepath.Join(projectpath.Root, "_support/test/shards")
	client := zoekt.NewZoektClient(&zoekt.Options{
		IndexDir: dir,
		ID:       ExistingID,
	})

	result, ok, err := client.GetCurrentSHA()

	require.NoError(t, err)

	expectedSHA := "5f6ffd6461ba03e257d24eed4f1f33a7ee3c2784"
	require.Equal(t, expectedSHA, result, "The two SHA should be the same.")
	require.True(t, ok)

	client = zoekt.NewZoektClient(&zoekt.Options{
		IndexDir: dir,
		ID:       NonExistentID,
	})

	_, ok, err = client.GetCurrentSHA()

	require.NoError(t, err)
	require.False(t, ok)
}

func TestGetCurrentSHAWithMetaProjectID(t *testing.T) {
	dir := filepath.Join(projectpath.Root, "_support/test/shards")
	client := zoekt.NewZoektClient(&zoekt.Options{
		IndexDir: dir,
		ID:       ExistingID,
		Metadata: map[string]string{"project_id": fmt.Sprintf("%d", ExistingID)},
	})

	result, ok, err := client.GetCurrentSHA()

	require.NoError(t, err)
	require.NotNil(t, result, "Expected a non-nil result")

	expectedSHA := "5f6ffd6461ba03e257d24eed4f1f33a7ee3c2784"
	require.Equal(t, expectedSHA, result, "The two SHA should be the same.")
	require.True(t, ok)

	client = zoekt.NewZoektClient(&zoekt.Options{
		IndexDir: dir,
		ID:       NonExistentID,
		Metadata: map[string]string{"project_id": ProjectID},
	})

	_, ok, err = client.GetCurrentSHA()

	require.NoError(t, err)
	require.False(t, ok)
}

func TestGetShardPrefix(t *testing.T) {
	cases := []struct {
		id        uint64
		projectID string
		expected  string
	}{
		{10, "12345", "12345"},
		{10, "", "10"},
		{0, "", "0"},
		{42, "999", "999"},
	}

	for _, c := range cases {
		opts := &zoekt.Options{
			ID: c.id,
		}
		if c.projectID != "" {
			opts.Metadata = map[string]string{"project_id": c.projectID}
		}
		prefix := zoekt.GetShardPrefix(opts)
		require.Equal(t, c.expected, prefix, "id: %d, projectID: %s", c.id, c.projectID)
	}
}

func TestMain(m *testing.M) {
	goleak.VerifyTestMain(m)
}
