package webserver_test_helper //nolint:staticcheck

import (
	"bytes"
	"context"
	"encoding/json"
	"errors"
	"fmt"
	"log/slog"
	"net"
	"net/http"
	"net/url"
	"os"
	"os/exec"
	"path/filepath"
	"runtime"
	"time"

	"gitlab.com/gitlab-org/gitlab-zoekt-indexer/internal/authentication"
)

const BinaryName = "gitlab-zoekt"

type WebserverHelper struct {
	indexDir    string
	port        int
	secretFile  string
	secretBytes []byte
}

type searchQuery struct {
	Q       string            `json:"q"`
	Opts    searchOpts        `json:"opts"`
	RepoIDs []uint64          `json:"repoIDs"`
	Meta    map[string]string `json:"meta,omitempty"`
}

type searchOpts struct {
	TotalMaxMatchCount int `json:"totalMaxMatchCount"`
	NumContextLines    int `json:"numContextLines"`
}

type SearchResponse struct {
	Result searchResponseResult `json:"result"`
}

type searchResponseResult struct {
	FileCount  int                   `json:"fileCount"`
	MatchCount int                   `json:"matchCount"`
	Files      []searchResponseFiles `json:"files"`
}

type searchResponseFiles struct {
	FileName string   `json:"fileName"`
	Branches []string `json:"branches"`
	Checksum []byte   `json:"checksum"`
}

func (h *SearchResponse) FileNames() []string {
	files := make([]string, 0, len(h.Result.Files))

	for _, f := range h.Result.Files {
		files = append(files, f.FileName)
	}

	return files
}

// findProjectDir returns the project root directory.
// Since we know this file is in internal/webserver_test_helper,
// we can simply go up two directories to find the project root.
func findProjectDir() string {
	// Get the current file's directory
	_, filename, _, ok := runtime.Caller(0)
	if !ok {
		slog.Debug("Failed to get current file path")
		return ""
	}

	// From internal/webserver_test_helper, go up two directories to project root
	dir := filepath.Dir(filename)                                 // dir of this file
	projectRoot, err := filepath.Abs(filepath.Join(dir, "../..")) // go up two levels
	if err != nil {
		slog.Debug("Failed to determine project root", "error", err)
		return ""
	}

	return projectRoot
}

func NewWebserverHelper(indexDir string) *WebserverHelper {
	port := findAvailablePort()

	// Create a temporary secret file for testing
	secretBytes := []byte("test-secret-key")
	secretFile := filepath.Join(os.TempDir(), fmt.Sprintf("zoekt-secret-%d", port))
	err := os.WriteFile(secretFile, secretBytes, 0600)
	if err != nil {
		panic(fmt.Errorf("failed to create secret file: %w", err))
	}

	return &WebserverHelper{
		indexDir:    indexDir,
		port:        port,
		secretFile:  secretFile,
		secretBytes: secretBytes,
	}
}

func (h *WebserverHelper) Port() int {
	return h.port
}

func findAvailablePort() int {
	lc := net.ListenConfig{}
	ln, err := lc.Listen(context.Background(), "tcp", "127.0.0.1:0")
	if err != nil {
		panic(err)
	}
	port := ln.Addr().(*net.TCPAddr).Port
	_ = ln.Close() // Ignore close error as we've already obtained the port number
	return port
}

func (h *WebserverHelper) listen() string {
	return fmt.Sprintf(":%d", h.port)
}

func (h *WebserverHelper) baseURL() string {
	return fmt.Sprintf("http://localhost:%d", h.port)
}

func (h *WebserverHelper) healthURL() string {
	res, err := url.JoinPath(h.baseURL(), "healthz")
	if err != nil {
		panic(err)
	}

	return res
}

func (h *WebserverHelper) searchURL() string {
	res, err := url.JoinPath(h.baseURL(), "api", "search")

	if err != nil {
		panic(err)
	}

	return res
}

func (h *WebserverHelper) isReady() bool {
	res, err := http.Head(h.healthURL()) //nolint:bodyclose,noctx

	if err != nil {
		return false
	}

	return res.StatusCode == 200
}

func (h *WebserverHelper) Start() (func(), error) {
	// First, try to find the binary in the project's bin directory
	projectDir := findProjectDir()
	binaryPath := BinaryName

	if projectDir != "" {
		candidatePath := fmt.Sprintf("%s/bin/%s", projectDir, BinaryName)
		if _, err := os.Stat(candidatePath); err == nil {
			binaryPath = candidatePath
		}
	}

	// If not found in project bin, check PATH
	if binaryPath == BinaryName {
		path, err := exec.LookPath(BinaryName)
		if err != nil {
			return func() {}, fmt.Errorf("binary %s not found in <PROJECT_DIR>/bin or PATH: %w", BinaryName, err)
		}
		binaryPath = path
	}

	cmd := exec.CommandContext(context.Background(), binaryPath, "webserver", "-index_dir", h.indexDir, "-listen", h.listen(), "-secret_path", h.secretFile) //nolint:gosec
	cmd.Stdout = os.Stdout
	cmd.Stderr = os.Stderr

	if err := cmd.Start(); err != nil {
		return func() {}, err
	}

	timeout := time.Now().Add(30 * time.Second)
	for !h.isReady() {
		if time.Now().After(timeout) {
			return func() {}, fmt.Errorf("webserver failed to start on port %d within 30 seconds", h.port)
		}
		time.Sleep(100 * time.Millisecond)
	}

	cancelFunc := func() {
		proc, err := os.FindProcess(cmd.Process.Pid)

		if err != nil {
			slog.Error("find process error", "err", err)
		}

		if err := proc.Kill(); err != nil {
			slog.Error("kill error", "err", err)
		}

		if err := cmd.Wait(); err != nil {
			slog.Error("wait() error", "err", err)
		}

		// Clean up the temporary secret file
		if err := os.Remove(h.secretFile); err != nil {
			slog.Error("failed to remove secret file", "err", err)
		}
	}

	return cancelFunc, nil
}

func (h *WebserverHelper) Search(repoID uint64, term string) (SearchResponse, error) {
	query := searchQuery{
		Q: term,
		Opts: searchOpts{
			TotalMaxMatchCount: 20,
			NumContextLines:    0,
		},
		Meta: map[string]string{
			"key":   "project_id",
			"value": fmt.Sprintf("%d", repoID),
		},
	}

	queryJSON, err := json.Marshal(query)
	if err != nil {
		return SearchResponse{}, err
	}

	// Generate JWT token for authentication
	auth := authentication.NewAuth(authentication.GitlabIssuer, 5*time.Minute, h.secretBytes)
	token, err := auth.GenerateJWT()
	if err != nil {
		return SearchResponse{}, fmt.Errorf("failed to generate JWT token: %w", err)
	}

	req, err := http.NewRequestWithContext(context.Background(), "POST", h.searchURL(), bytes.NewBuffer(queryJSON))
	if err != nil {
		return SearchResponse{}, err
	}

	req.Header.Set("Content-Type", "application/json")
	req.Header.Set(authentication.GitlabZoektAPIRequestHeader, "Bearer "+token)

	client := &http.Client{}
	res, err := client.Do(req)

	if err != nil {
		return SearchResponse{}, err
	}
	defer func() {
		if closeErr := res.Body.Close(); closeErr != nil {
			slog.Error("failed to close response body", "err", closeErr)
		}
	}()

	if res.StatusCode != http.StatusOK {
		return SearchResponse{}, errors.New("search failed with non OK")
	}

	var result SearchResponse
	dec := json.NewDecoder(res.Body)
	if err := dec.Decode(&result); err != nil {
		return SearchResponse{}, err
	}

	return result, nil
}
