package task_request_test

import (
	"context"
	"encoding/json"
	"fmt"
	"io"
	"net/http"
	"net/http/httptest"
	"net/url"
	"testing"

	"github.com/stretchr/testify/require"
	"gitlab.com/gitlab-org/gitlab-zoekt-indexer/internal/indexing_lock"
	"gitlab.com/gitlab-org/gitlab-zoekt-indexer/internal/task_request"
	"go.uber.org/goleak"
)

const (
	nodeName  = "nodeName"
	nodeUUID  = "3869fe21-36d1"
	version   = "dev"
	searchURL = "http://search.url"
	selfURL   = "http://self.url"
)

func TestNewTaskRequestTimerWithSecret(t *testing.T) {
	indexDir := t.TempDir()
	secret := []byte("secret")

	timer, err := task_request.NewTaskRequestTimer(&task_request.NewTaskRequestTimerParams{
		IndexDir:  indexDir,
		NodeName:  nodeName,
		SelfURL:   selfURL,
		SearchURL: searchURL,
		GitlabURL: "http://gitlab.url",
		Services:  []string{"zoekt"},
		Secret:    secret,
	})

	require.NoError(t, err)
	require.NotNil(t, timer)
}

func TestSendRequest(t *testing.T) {
	indexDir := t.TempDir()
	secret := []byte("secret")

	var requestedURL *url.URL
	var requestedBody any

	svr := httptest.NewServer(http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
		requestedURL = r.URL
		buf, _ := io.ReadAll(r.Body)
		err := json.Unmarshal(buf, &requestedBody)
		if err != nil {
			panic(err)
		}
	}))
	defer svr.Close()

	indexingLock := indexing_lock.NewIndexingLock()
	locked := indexingLock.TryLock(55)
	require.True(t, locked)

	requestTimer, err := task_request.NewTaskRequestTimer(&task_request.NewTaskRequestTimerParams{
		IndexDir:     indexDir,
		NodeName:     nodeName,
		Version:      version,
		SelfURL:      selfURL,
		SearchURL:    searchURL,
		GitlabURL:    svr.URL,
		Services:     []string{"zoekt"},
		Secret:       secret,
		NodeUUID:     nodeUUID,
		IndexingLock: indexingLock,
	})

	require.NoError(t, err)

	_, requestErr := requestTimer.SendRequest(context.Background())
	require.NoError(t, requestErr)
	require.NoError(t, err)

	expectedPath := fmt.Sprintf("/api/v4/internal/search/zoekt/%s/heartbeat", url.PathEscape(nodeUUID))
	expectedTaskCount := "1"
	expectedConcurrency := "0" // runtime.GOMAXPROCS(0) returns zero when executed from tests
	expectedSchemaVersion := float64(2531)

	require.Equal(t, expectedPath, requestedURL.Path)
	require.Equal(t, selfURL, requestedBody.(map[string]interface{})["node.url"])
	require.Equal(t, searchURL, requestedBody.(map[string]interface{})["node.search_url"])
	require.Equal(t, nodeName, requestedBody.(map[string]interface{})["node.name"])
	require.Equal(t, version, requestedBody.(map[string]interface{})["node.version"])
	require.Equal(t, expectedConcurrency, requestedBody.(map[string]interface{})["node.concurrency"])
	require.Equal(t, expectedTaskCount, requestedBody.(map[string]interface{})["node.task_count"])
	require.Equal(t, expectedSchemaVersion, requestedBody.(map[string]interface{})["node.schema_version"])
	require.NotEmpty(t, requestedBody.(map[string]interface{})["disk.all"])
	require.NotEmpty(t, requestedBody.(map[string]interface{})["disk.free"])
	require.NotEmpty(t, requestedBody.(map[string]interface{})["disk.indexed"])
	require.NotEmpty(t, requestedBody.(map[string]interface{})["disk.used"])
}

func TestMain(m *testing.M) {
	// Ignore glog goleaks since it is indirect
	goleak.VerifyTestMain(m, goleak.IgnoreTopFunction("github.com/golang/glog.(*fileSink).flushDaemon"))
}
