package server

import (
	"context"

	"github.com/prometheus/client_golang/prometheus"
	"gitlab.com/gitlab-org/gitlab-zoekt-indexer/internal/authentication"
	"gitlab.com/gitlab-org/gitlab-zoekt-indexer/internal/callback"
	"gitlab.com/gitlab-org/gitlab-zoekt-indexer/internal/indexing_lock"
	"gitlab.com/gitlab-org/gitlab-zoekt-indexer/internal/profiler"
)

type DefaultIndexBuilder struct {
	IndexDir string
}

type DeleteRequest struct {
	RepoID   uint64                   `json:"RepoID"`
	Callback *callback.CallbackParams `json:"Callback"`
}

type GitalyConnectionInfo struct {
	Address string `json:"Address"`
	Token   string `json:"Token"`
	Storage string `json:"Storage"`
	Path    string `json:"Path"`
}

type IndexRequest struct {
	Timeout              string                   `json:"Timeout"`
	RepoID               uint64                   `json:"RepoID"`
	GitalyConnectionInfo *GitalyConnectionInfo    `json:"GitalyConnectionInfo"`
	FileSizeLimit        int                      `json:"FileSizeLimit"`
	FileCountLimit       uint32                   `json:"FileCountLimit"`
	Callback             *callback.CallbackParams `json:"Callback"`
	Force                bool                     `json:"Force"`
	MissingRepo          bool                     `json:"MissingRepo"`
	Parallelism          int                      `json:"Parallelism"`
	TrigramMax           int                      `json:"TrigramMax"` // Validated by Rails to be positive
	Metadata             map[string]string        `json:"Metadata"`
}

type IndexServerMode interface {
	Init(version string)
	Services() []string
}

type ZoektServerMode struct {
	profiler profiler.Profiler
}

type IndexServer struct {
	IndexServerMode
	PathPrefix           string
	IndexBuilder         indexBuilder
	CallbackAPI          callback.CallbackAPI
	promRegistry         *prometheus.Registry
	metricsRequestsTotal *prometheus.CounterVec
	IndexingLock         *indexing_lock.IndexingLock
	Auth                 *authentication.Auth
	JWTEnabled           bool
}

type indexBuilder interface {
	DeleteRepository(DeleteRequest, callback.CallbackFunc, *indexing_lock.IndexingLock) error
	GetIndexDir() string
	IndexRepository(context.Context, IndexRequest, callback.CallbackFunc) error
}
