package secretreader_test

import (
	"os"
	"path/filepath"
	"testing"

	"github.com/stretchr/testify/require"
	"gitlab.com/gitlab-org/gitlab-zoekt-indexer/internal/secretreader"
)

func TestReadSecret(t *testing.T) {
	tests := []struct {
		name           string
		fileContent    []byte
		expectedSecret string
		expectError    bool
	}{
		{
			name:           "clean secret without whitespace",
			fileContent:    []byte("test-secret"),
			expectedSecret: "test-secret",
			expectError:    false,
		},
		{
			name:           "secret with spaces",
			fileContent:    []byte("  test-secret  "),
			expectedSecret: "test-secret",
			expectError:    false,
		},
		{
			name:           "secret with tabs",
			fileContent:    []byte("\ttest-secret\t"),
			expectedSecret: "test-secret",
			expectError:    false,
		},
		{
			name:           "secret with newlines",
			fileContent:    []byte("\ntest-secret\n"),
			expectedSecret: "test-secret",
			expectError:    false,
		},
		{
			name:           "secret with mixed whitespace",
			fileContent:    []byte("\n\t test-secret \t\n"),
			expectedSecret: "test-secret",
			expectError:    false,
		},
		{
			name:           "only whitespace",
			fileContent:    []byte("\n\t \n"),
			expectedSecret: "",
			expectError:    true,
		},
		{
			name:           "empty file",
			fileContent:    []byte(""),
			expectedSecret: "",
			expectError:    true,
		},
	}

	for _, tt := range tests {
		t.Run(tt.name, func(t *testing.T) {
			// Create a temporary file with the test content
			tempDir := t.TempDir()
			secretPath := filepath.Join(tempDir, "secret.txt")

			err := os.WriteFile(secretPath, tt.fileContent, 0600)
			require.NoError(t, err, "Failed to write secret content to file")

			// Call the function under test
			secret, err := secretreader.ReadSecret(secretPath)

			if tt.expectError {
				require.ErrorIs(t, err, secretreader.ErrEmptySecret)

				return
			}

			require.NoError(t, err)
			require.Equal(t, tt.expectedSecret, string(secret),
				"Secret was not processed correctly for case: %s", tt.name)
		})
	}
}

func TestReadSecretWithInvalidPath(t *testing.T) {
	// Test with a non-existent file
	tempDir := t.TempDir()
	nonExistentPath := filepath.Join(tempDir, "non-existent.txt")

	_, err := secretreader.ReadSecret(nonExistentPath)
	require.Error(t, err)
	require.Contains(t, err.Error(), "no such file or directory")
}
