// Package secretreader provides functionality for securely reading secret values from files.
// It handles file reading, whitespace trimming, and validation to ensure that secrets
// are properly loaded. The package implements error handling for common issues such as
// missing files or empty content.
package secretreader

import (
	"bytes"
	"errors"
	"fmt"
	"os"
	"path/filepath"
)

// ErrEmptySecret is returned when a secret file is empty or contains only whitespace
var ErrEmptySecret = errors.New("secret is empty after trimming whitespace")

// ReadSecret reads and trims a secret from a file
func ReadSecret(path string) ([]byte, error) {
	f, err := os.ReadFile(filepath.Clean(path))
	if err != nil {
		return nil, fmt.Errorf("failed to read secret file %q: %w", path, err)
	}

	secret := bytes.TrimSpace(f)
	if len(secret) == 0 {
		return nil, fmt.Errorf("failed to read secret file %q: %w", path, ErrEmptySecret)
	}

	return secret, nil
}
