package search

import (
	"net/http"
	"sync"
	"time"

	"github.com/sourcegraph/zoekt"
	"google.golang.org/grpc"
)

type Searcher struct {
	Client *http.Client

	GrpcConns     map[string]*grpc.ClientConn // Map of endpoint to gRPC connection
	grpcMutex     sync.RWMutex
	TLSServerName string
}

type SearchRequest struct {
	TimeoutString              string        `json:"timeout"`
	Timeout                    time.Duration `json:"-"` // Parsed timeout for internal use
	NumContextLines            uint32        `json:"num_context_lines"`
	MaxFileMatchWindow         uint32        `json:"max_file_match_window"`
	MaxFileMatchResults        uint32        `json:"max_file_match_results"`
	MaxLineMatchWindow         uint32        `json:"max_line_match_window"`
	MaxLineMatchResults        uint32        `json:"max_line_match_results"`
	MaxLineMatchResultsPerFile uint32        `json:"max_line_match_results_per_file"`
	ForwardTo                  []Search      `json:"forward_to"`
}

type Search struct {
	Endpoint string           `json:"endpoint"`
	Options  *EndpointOptions `json:"options,omitempty"`
	Query    Query            `json:"query"`
}

type EndpointOptions struct {
	Headers map[string]string `json:"headers,omitempty"`
}

type RawSearchRequest struct {
	Version uint8 `json:"version,omitempty"` // Version of the request format

	Timeout time.Duration `json:"-"` // Parsed timeout for internal use

	// V2 fields (improved format)
	TimeoutString              string   `json:"timeout,omitempty"`
	NumContextLines            uint32   `json:"num_context_lines,omitempty"`
	MaxFileMatchWindow         uint32   `json:"max_file_match_window,omitempty"`
	MaxFileMatchResults        uint32   `json:"max_file_match_results,omitempty"`
	MaxLineMatchWindow         uint32   `json:"max_line_match_window,omitempty"`
	MaxLineMatchResults        uint32   `json:"max_line_match_results,omitempty"`
	MaxLineMatchResultsPerFile uint32   `json:"max_line_match_results_per_file,omitempty"`
	ForwardTo                  []Search `json:"forward_to,omitempty"`

	// V1 fields (legacy support)
	ZoektSearchRequest
	TimeoutStringV1 string            `json:"Timeout,omitempty"`
	ForwardToV1     []Conn            `json:"ForwardTo,omitempty"`
	HeadersV1       map[string]string `json:"-"` // Computed headers for internal use
}

type Conn struct {
	Endpoint string   `json:"Endpoint"`
	RepoIds  []uint32 `json:"RepoIds"`
}

type SearchResult struct {
	Result   Result          `json:"Result"`
	Error    string          `json:"Error"`
	Failures []SearchFailure `json:"Failures"`
	TimedOut bool            `json:"TimedOut"`
}

type SearchFailure struct {
	Error    string `json:"Error"`
	Endpoint string `json:"Endpoint"`
}

type Result struct {
	Files        []zoekt.FileMatch `json:"Files"`
	FileCount    uint32            `json:"FileCount"`
	MatchCount   uint32            `json:"MatchCount"`
	NgramMatches uint32            `json:"NgramMatches"`

	TotalFileMatchCount uint64 `json:"TotalFileMatchCount"` // synonymous with FileCount
	TotalLineMatchCount uint64 `json:"TotalLineMatchCount"` // synonymous with MatchCount
	FileMatchCount      uint64 `json:"FileMatchCount"`
	LineMatchCount      uint64 `json:"LineMatchCount"`
}

type ZoektSearchRequest struct {
	Query   string        `json:"Q"`
	Options SearchOptions `json:"Opts"`
	RepoIds []uint64      `json:"RepoIds"`
}

type SearchOptions struct {
	TotalMaxMatchCount uint32 `json:"TotalMaxMatchCount"`
	NumContextLines    uint32 `json:"NumContextLines"`

	MaxFileMatchWindow         uint32 `json:"MaxFileMatchWindow"` // Used for post-filtering file count results, NOT passed to Zoekt
	MaxFileMatchResults        uint32 `json:"MaxFileMatchResults"`
	MaxLineMatchWindow         uint32 `json:"MaxLineMatchWindow"`
	MaxLineMatchResults        uint32 `json:"MaxLineMatchResults"`
	MaxLineMatchResultsPerFile uint32 `json:"MaxLineMatchResultsPerFile"`
}

type ZoektResponse struct {
	Result   zoekt.SearchResult
	Error    error
	Endpoint string
}

type EndpointInfo struct {
	Original string // The original endpoint string
	Protocol string // The protocol (e.g., "http" or "https")
	HostPort string // The host:port portion (e.g., "example.com:8080")
	Path     string // The path portion with leading slash (e.g., "/api/search")
}
