package search

import (
	"encoding/json"
	"fmt"
	"io"
	"net/http"
	"net/http/httptest"
	"strings"
	"testing"
	"time"

	"github.com/sourcegraph/zoekt"
	proto "github.com/sourcegraph/zoekt/grpc/protos/zoekt/webserver/v1"
	"github.com/stretchr/testify/require"
	"google.golang.org/grpc"
)

func TestNewSearchRequestInvalidForwardTo(t *testing.T) {
	t.Parallel()
	reqBody := `{
		"Q": "test query",
		"Opts": {"TotalMaxMatchCount": 10, "NumContextLines": 2},
		"ForwardTo": [{"RepoIds": [1, 2, 3]}]
	}`
	req := httptest.NewRequest("POST", "/search", strings.NewReader(reqBody))
	req.Header.Set("Content-Type", "application/json")
	_, err := NewSearchRequest(req)
	require.Error(t, err)
	require.Equal(t, "forward-to endpoint is empty", err.Error())
}

func TestNewSearchRequestNoRepoIds(t *testing.T) {
	t.Parallel()
	reqBody := `{
		"Q": "test query",
		"Opts": {"TotalMaxMatchCount": 10, "NumContextLines": 2},
		"ForwardTo": [{"Endpoint": "http://example.com"}]
	}`
	req := httptest.NewRequest("POST", "/search", strings.NewReader(reqBody))
	req.Header.Set("Content-Type", "application/json")
	_, err := NewSearchRequest(req)
	require.Error(t, err)
	require.Regexp(t, "no repo IDs specified for forward-to connection", err.Error())
}

func TestSearchResultSorting_GRPC(t *testing.T) {
	t.Parallel()

	files1 := []zoekt.FileMatch{
		{FileName: "file1.go", Repository: "repo1", Score: 0.5, Checksum: []byte{0x1, 0x2, 0x3}},
		{FileName: "file2.go", Repository: "repo1", Score: 0.8, Checksum: []byte{0x4, 0x5, 0x6}},
	}

	files2 := []zoekt.FileMatch{
		{FileName: "file3.go", Repository: "repo2", Score: 0.9, Checksum: []byte{0x7, 0x8, 0x9}},
		{FileName: "file4.go", Repository: "repo2", Score: 0.3, Checksum: []byte{0x3, 0x1, 0x2}},
	}

	conn1, endpoint1 := StartBufconnServer(t, ToProtoFileMatches(files1), &proto.Stats{})
	defer conn1.Close()
	conn2, endpoint2 := StartBufconnServer(t, ToProtoFileMatches(files2), &proto.Stats{})
	defer conn2.Close()

	searcher := &Searcher{
		GrpcConns: map[string]*grpc.ClientConn{
			GetHostFromEndpoint(endpoint1): conn1,
			GetHostFromEndpoint(endpoint2): conn2,
		},
	}

	// Build JSON request body
	body := fmt.Sprintf(`{
		"Q": "test",
		"Grpc": true,
		"Opts": {
			"TotalMaxMatchCount": 10,
			"NumContextLines": 2
		},
		"Timeout": "1s",
		"ForwardTo": [
			{"Endpoint": "%s", "RepoIds": [1]},
			{"Endpoint": "%s", "RepoIds": [2]}
		]
	}`, endpoint1, endpoint2)

	req := &http.Request{
		Body:   io.NopCloser(strings.NewReader(body)),
		Header: make(http.Header),
	}
	req.Header.Set("Content-Type", "application/json")

	resp, err := searcher.Search(req)
	require.NoError(t, err)
	require.Len(t, resp.Result.Files, 4)

	require.Equal(t, "file3.go", resp.Result.Files[0].FileName)
	require.Equal(t, "file2.go", resp.Result.Files[1].FileName)
	require.Equal(t, "file1.go", resp.Result.Files[2].FileName)
	require.Equal(t, "file4.go", resp.Result.Files[3].FileName)

}

func TestSearchWithPartialFailure_GRPC(t *testing.T) {
	t.Parallel()

	successFiles := ToProtoFileMatches([]zoekt.FileMatch{
		{FileName: "success.go", Score: 1.0},
	})
	conn1, endpoint1 := StartBufconnServer(t, successFiles, &proto.Stats{})
	defer conn1.Close()

	conn2, endpoint2 := StartFailingBufconnServer(t, "Internal Server Error")
	defer conn2.Close()

	searcher := &Searcher{
		GrpcConns: map[string]*grpc.ClientConn{
			GetHostFromEndpoint(endpoint1): conn1,
			GetHostFromEndpoint(endpoint2): conn2,
		},
	}

	body := fmt.Sprintf(`{
		"Q": "test",
		"Grpc": true,
		"Opts": {"TotalMaxMatchCount": 10, "NumContextLines": 2},
		"Timeout": "1s",
		"ForwardTo": [
			{"Endpoint": "%s", "RepoIds": [1]},
			{"Endpoint": "%s", "RepoIds": [2]}
		]
	}`, endpoint1, endpoint2)

	req := &http.Request{
		Body:   io.NopCloser(strings.NewReader(body)),
		Header: make(http.Header),
	}
	req.Header.Set("Content-Type", "application/json")

	// Call real search
	result, err := searcher.Search(req)

	require.NoError(t, err)
	require.NotNil(t, result)

	require.Len(t, result.Result.Files, 1)
	require.Equal(t, "success.go", result.Result.Files[0].FileName)

	require.Len(t, result.Failures, 1)
	require.Contains(t, result.Failures[0].Error, "Internal Server Error")
}

func TestSearchAllServersFail_GRPC(t *testing.T) {
	t.Parallel()

	conn1, endpoint1 := StartFailingBufconnServer(t, "Internal Server Error")
	defer conn1.Close()

	conn2, endpoint2 := StartFailingBufconnServer(t, "Bad Gateway")
	defer conn2.Close()

	searcher := &Searcher{
		GrpcConns: map[string]*grpc.ClientConn{
			GetHostFromEndpoint(endpoint1): conn1,
			GetHostFromEndpoint(endpoint2): conn2,
		},
	}

	body := fmt.Sprintf(`{
		"Q": "test",
		"Grpc": true,
		"Opts": {"TotalMaxMatchCount": 10, "NumContextLines": 2},
		"Timeout": "1s",
		"ForwardTo": [
			{"Endpoint": "%s", "RepoIds": [1]},
			{"Endpoint": "%s", "RepoIds": [2]}
		]
	}`, endpoint1, endpoint2)

	req := &http.Request{
		Body:   io.NopCloser(strings.NewReader(body)),
		Header: make(http.Header),
	}
	req.Header.Set("Content-Type", "application/json")

	result, err := searcher.Search(req)

	require.Error(t, err)
	require.Nil(t, result)

	require.Contains(t, err.Error(), "all searches failed")
	require.Contains(t, err.Error(), "Internal Server Error")
	require.Contains(t, err.Error(), "Bad Gateway")
}

func TestSearchTimeoutWithNoResults(t *testing.T) {
	t.Parallel()
	// Create a mock server that simulates a slow response with no results
	slowServer := httptest.NewServer(http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
		time.Sleep(200 * time.Millisecond)
		w.Header().Set("Content-Type", "application/json")
		json.NewEncoder(w).Encode(ZoektResponse{
			Result: zoekt.SearchResult{
				Files: []zoekt.FileMatch{},
			},
		})
	}))
	defer slowServer.Close()

	// Create a search request with a very short timeout
	req := &http.Request{
		Body: io.NopCloser(strings.NewReader(`{
			"Q": "test",
			"Opts": {"TotalMaxMatchCount": 10, "NumContextLines": 2},
			"Timeout": "100ms",
			"ForwardTo": [{"Endpoint": "` + slowServer.URL + `", "RepoIds": [1, 2, 3]}]
		}`)),
		Header: make(http.Header),
	}
	req.Header.Set("Content-Type", "application/json")

	result, err := testSearcher().Search(req)

	// Check that the search returned an error
	require.Error(t, err)
	require.Nil(t, result)

	// Check that the error message indicates a timeout
	require.EqualError(t, err, "multi node search failed: search timed out")
}

func TestNewSearchRequestDefaultTimeout(t *testing.T) {
	t.Parallel()
	// Create a request with no timeout specified
	reqBody := `{
		"Q": "test query",
		"Opts": {"TotalMaxMatchCount": 10, "NumContextLines": 2},
		"ForwardTo": [{"Endpoint": "http://example.com", "RepoIds": [1, 2, 3]}]
	}`
	req := httptest.NewRequest("POST", "/search", strings.NewReader(reqBody))
	req.Header.Set("Content-Type", "application/json")
	searchReq, err := NewSearchRequest(req)
	require.NoError(t, err)
	require.Equal(t, defaultSearchTimeout, searchReq.TimeoutString)
}

func TestNewSearchRequestDefaultMaxResults(t *testing.T) {
	t.Parallel()
	// Create a request with no TotalMaxMatchCount specified
	reqBody := `{
		"Q": "test query",
		"Opts": {"NumContextLines": 2
		},
		"ForwardTo": [{"Endpoint": "http://example.com", "RepoIds": [1, 2, 3]}]
	}`
	req := httptest.NewRequest("POST", "/search", strings.NewReader(reqBody))
	req.Header.Set("Content-Type", "application/json")
	searchReq, err := NewSearchRequest(req)
	require.NoError(t, err)
	require.Equal(t, defaultMaxLineMatchWindow, searchReq.MaxLineMatchWindow)
	require.Equal(t, uint32(0), searchReq.MaxFileMatchWindow)
	require.Equal(t, uint32(0), searchReq.MaxFileMatchResults)
	require.Equal(t, uint32(0), searchReq.MaxLineMatchResults)
	require.Equal(t, uint32(0), searchReq.MaxLineMatchResultsPerFile)
}

func TestNewSearchRequestCopiesBasicAuthHeaders(t *testing.T) {
	t.Parallel()
	// Create a request with basic auth headers
	reqBody := `{
		"Q": "test query",
		"Opts": {"TotalMaxMatchCount": 10, "NumContextLines": 2},
		"ForwardTo": [{"Endpoint": "http://example.com", "RepoIds": [1, 2, 3]}]
	}`
	req := httptest.NewRequest("POST", "/search", strings.NewReader(reqBody))
	req.Header.Set("Content-Type", "application/json")
	req.SetBasicAuth("username", "password")

	searchReq, err := NewSearchRequest(req)
	require.NoError(t, err)

	// Check if the basic auth header was copied
	for _, conn := range searchReq.ForwardTo {
		authHeader, exists := conn.Options.Headers["Authorization"]
		require.True(t, exists)
		require.Equal(t, "Basic dXNlcm5hbWU6cGFzc3dvcmQ=", authHeader)
		require.Equal(t, "http://example.com", conn.Endpoint)
	}
}

func TestSearchWithEmptyQuery(t *testing.T) {
	t.Parallel()
	// Create a search request with an empty query
	req := &http.Request{
		Body: io.NopCloser(strings.NewReader(`{
			"Q": "",
			"Opts": {"TotalMaxMatchCount": 10, "NumContextLines": 2},
			"Timeout": "1s",
			"ForwardTo": [{"Endpoint": "http://example.com", "RepoIds": [1, 2, 3]}]
		}`)),
		Header: make(http.Header),
	}
	req.Header.Set("Content-Type", "application/json")

	result, err := testSearcher().Search(req)

	// Check that the search returned an error
	require.Error(t, err)
	require.Nil(t, result)

	// Check that the error message indicates an empty query
	require.Regexp(t, "search query is empty", err.Error())
}

func TestSearchWithNoForwardToConnections(t *testing.T) {
	t.Parallel()
	// Create a search request with no forward to connections
	req := &http.Request{
		Body: io.NopCloser(strings.NewReader(`{
				"Q": "test",
				"Opts": {"TotalMaxMatchCount": 10, "NumContextLines": 2},
				"Timeout": "1s",
				"ForwardTo": []
			}`)),
		Header: make(http.Header),
	}
	req.Header.Set("Content-Type", "application/json")

	result, err := testSearcher().Search(req)

	// Check that the search returned an error
	require.Error(t, err)
	require.Nil(t, result)

	// Check that the error message indicates an empty query
	require.EqualError(t, err, "error building search request: no forward-to connections specified")
}

func TestSearchCannotParseRequestBody(t *testing.T) {
	t.Parallel()
	// Create a search request with a field that can't be marshaled
	req := &http.Request{
		Body: io.NopCloser(strings.NewReader(`{
			"Q": "test",
			"Opts": {"TotalMaxMatchCount": 10, "NumContextLines": 2},
			"Timeout": "1s",
			"ForwardTo": [{"Endpoint": "http://example.com", "RepoIds": [1, 2, 3]}],
			"UnmarshalableField": func() {}
		}`)),
		Header: make(http.Header),
	}
	req.Header.Set("Content-Type", "application/json")

	result, err := testSearcher().Search(req)

	// Check that the search returned an error
	require.Error(t, err)
	require.Nil(t, result)

	// Check that the error message indicates a parsing problem
	require.Contains(t, err.Error(), "failed to parse request body")
}

func TestSearchMaxLineMatchResults_GRPC(t *testing.T) {
	t.Parallel()

	// Helper to create files with line matches
	makeFile := func(file string, score float64, numLines int) zoekt.FileMatch {
		lines := make([]zoekt.LineMatch, numLines)
		for i := range lines {
			lines[i] = zoekt.LineMatch{Line: []byte(fmt.Sprintf("match line %d", i))}
		}
		return zoekt.FileMatch{
			FileName:    file,
			Score:       score,
			LineMatches: lines,
		}
	}

	// Create mock files
	files1 := []zoekt.FileMatch{
		makeFile("file1.go", 0.6, 3),
		makeFile("file2.go", 0.7, 3),
	}
	files2 := []zoekt.FileMatch{
		makeFile("file3.go", 0.9, 3),
	}

	conn1, endpoint1 := StartBufconnServer(t, ToProtoFileMatches(files1), &proto.Stats{FileCount: 2})
	defer conn1.Close()

	conn2, endpoint2 := StartBufconnServer(t, ToProtoFileMatches(files2), &proto.Stats{FileCount: 1})
	defer conn2.Close()

	searcher := &Searcher{
		GrpcConns: map[string]*grpc.ClientConn{
			GetHostFromEndpoint(endpoint1): conn1,
			GetHostFromEndpoint(endpoint2): conn2,
		},
	}

	body := fmt.Sprintf(`{
		"Q": "test",
		"Grpc": true,
		"Opts": {
			"TotalMaxMatchCount": 100,
			"MaxLineMatchResults": 4,
			"MaxLineMatchResultsPerFile": 2,
			"NumContextLines": 2
		},
		"Timeout": "1s",
		"ForwardTo": [
			{"Endpoint": "%s", "RepoIds": [1]},
			{"Endpoint": "%s", "RepoIds": [2]}
		]
	}`, endpoint1, endpoint2)

	req := &http.Request{
		Body:   io.NopCloser(strings.NewReader(body)),
		Header: make(http.Header),
	}
	req.Header.Set("Content-Type", "application/json")

	result, err := searcher.Search(req)
	require.NoError(t, err)
	require.NotNil(t, result)

	// Max total line matches (across all files)
	var matchCount int
	for _, f := range result.Result.Files {
		matchCount += len(f.LineMatches)
	}
	require.LessOrEqual(t, matchCount, 4)

	// Per-file max
	for _, f := range result.Result.Files {
		require.LessOrEqual(t, len(f.LineMatches), 2)
	}

	// Sorted by score
	if len(result.Result.Files) > 1 {
		for i := 1; i < len(result.Result.Files); i++ {
			require.GreaterOrEqual(t,
				result.Result.Files[i-1].Score,
				result.Result.Files[i].Score,
			)
		}
	}
}

func testSearcher() *Searcher {
	return NewSearcher(&http.Client{
		CheckRedirect: func(req *http.Request, via []*http.Request) error {
			return http.ErrUseLastResponse
		},
	})
}

func TestSearchDeduplicatesIdenticalFiles_GRPC(t *testing.T) {
	t.Parallel()

	// Create a file match with a fixed checksum and repository
	file := zoekt.FileMatch{
		FileName:   "dupe.go",
		Repository: "repo1",
		Score:      1.0,
		Checksum:   []byte("fixed-checksum"),
	}

	// Start a mock gRPC server that always returns the same file
	conn, endpoint := StartBufconnServer(t, ToProtoFileMatches([]zoekt.FileMatch{file}), &proto.Stats{})
	defer conn.Close()

	searcher := &Searcher{
		GrpcConns: map[string]*grpc.ClientConn{
			GetHostFromEndpoint(endpoint): conn,
		},
	}

	forwardTo := fmt.Sprintf(`{
	"endpoint": "%s",
	"query": {
		"and": {
			"children": [
				{"repo_ids": [1,2,19]},
				{"substring": {"pattern": "test", "case_sensitive": false, "content": true}}
			]
			}
		}
        }`, endpoint)

	// Specify the same forward_to twice
	body := fmt.Sprintf(`{
		"version": 2,
		"timeout": "30s",
		"num_context_lines": 20,
		"max_file_match_window": 1000,
		"max_file_match_results": 5,
		"max_line_match_window": 500,
		"max_line_match_results": 10,
		"max_line_match_results_per_file": 3,
		"forward_to": [%s, %s]
	}`, forwardTo, forwardTo)

	req := &http.Request{
		Body:   io.NopCloser(strings.NewReader(body)),
		Header: make(http.Header),
	}
	req.Header.Set("Content-Type", "application/json")

	resp, err := searcher.Search(req)
	require.NoError(t, err)
	require.NotNil(t, resp)
	require.Len(t, resp.Result.Files, 1, "Expected deduplication of identical results from same repository")
	require.Equal(t, "dupe.go", resp.Result.Files[0].FileName)
}

func TestSearchDoesNotDeduplicateFilesDifferentRepos_GRPC(t *testing.T) {
	t.Parallel()

	// Create files with same checksum but different repositories
	file1 := zoekt.FileMatch{
		FileName:   "same-file.go",
		Repository: "repo1",
		Score:      1.0,
		Checksum:   []byte("same-checksum"),
	}
	file2 := zoekt.FileMatch{
		FileName:   "same-file.go",
		Repository: "repo2",
		Score:      1.0,
		Checksum:   []byte("same-checksum"),
	}

	// Start mock gRPC servers returning files from different repos
	conn1, endpoint1 := StartBufconnServer(t, ToProtoFileMatches([]zoekt.FileMatch{file1}), &proto.Stats{})
	defer conn1.Close()
	conn2, endpoint2 := StartBufconnServer(t, ToProtoFileMatches([]zoekt.FileMatch{file2}), &proto.Stats{})
	defer conn2.Close()

	searcher := &Searcher{
		GrpcConns: map[string]*grpc.ClientConn{
			GetHostFromEndpoint(endpoint1): conn1,
			GetHostFromEndpoint(endpoint2): conn2,
		},
	}

	forwardTo1 := fmt.Sprintf(`{
	"endpoint": "%s",
	"query": {
		"and": {
			"children": [
				{"repo_ids": [1]},
				{"substring": {"pattern": "test", "case_sensitive": false, "content": true}}
			]
			}
		}
        }`, endpoint1)

	forwardTo2 := fmt.Sprintf(`{
	"endpoint": "%s",
	"query": {
		"and": {
			"children": [
				{"repo_ids": [2]},
				{"substring": {"pattern": "test", "case_sensitive": false, "content": true}}
			]
			}
		}
        }`, endpoint2)

	body := fmt.Sprintf(`{
		"version": 2,
		"timeout": "30s",
		"num_context_lines": 20,
		"max_file_match_window": 1000,
		"max_file_match_results": 5,
		"max_line_match_window": 500,
		"max_line_match_results": 10,
		"max_line_match_results_per_file": 3,
		"forward_to": [%s, %s]
	}`, forwardTo1, forwardTo2)

	req := &http.Request{
		Body:   io.NopCloser(strings.NewReader(body)),
		Header: make(http.Header),
	}
	req.Header.Set("Content-Type", "application/json")

	resp, err := searcher.Search(req)
	require.NoError(t, err)
	require.NotNil(t, resp)
	require.Len(t, resp.Result.Files, 2, "Expected NO deduplication of files from different repositories")
	require.Equal(t, "same-file.go", resp.Result.Files[0].FileName)
	require.Equal(t, "same-file.go", resp.Result.Files[1].FileName)
	require.NotEqual(t, resp.Result.Files[0].Repository, resp.Result.Files[1].Repository, "Files should be from different repositories")
}

func TestSearchDoesNotDeduplicateDifferentFilesSameRepos_GRPC(t *testing.T) {
	t.Parallel()

	// Create files with same checksum, same repository but different filename
	file1 := zoekt.FileMatch{
		FileName:   "file.go",
		Repository: "repo1",
		Score:      1.0,
		Checksum:   []byte("same-checksum"),
	}
	file2 := zoekt.FileMatch{
		FileName:   "file2.go",
		Repository: "repo1",
		Score:      0.5,
		Checksum:   []byte("same-checksum"),
	}

	// Start mock gRPC servers returning files from same repo
	conn1, endpoint1 := StartBufconnServer(t, ToProtoFileMatches([]zoekt.FileMatch{file1}), &proto.Stats{})
	defer conn1.Close()
	conn2, endpoint2 := StartBufconnServer(t, ToProtoFileMatches([]zoekt.FileMatch{file2}), &proto.Stats{})
	defer conn2.Close()

	searcher := &Searcher{
		GrpcConns: map[string]*grpc.ClientConn{
			GetHostFromEndpoint(endpoint1): conn1,
			GetHostFromEndpoint(endpoint2): conn2,
		},
	}

	forwardTo1 := fmt.Sprintf(`{
	"endpoint": "%s",
	"query": {
		"and": {
			"children": [
				{"repo_ids": [1]},
				{"substring": {"pattern": "test", "case_sensitive": false, "content": true}}
			]
			}
		}
        }`, endpoint1)

	forwardTo2 := fmt.Sprintf(`{
	"endpoint": "%s",
	"query": {
		"and": {
			"children": [
				{"repo_ids": [1]},
				{"substring": {"pattern": "test", "case_sensitive": false, "content": true}}
			]
			}
		}
        }`, endpoint2)

	body := fmt.Sprintf(`{
		"version": 2,
		"timeout": "30s",
		"num_context_lines": 20,
		"max_file_match_window": 1000,
		"max_file_match_results": 5,
		"max_line_match_window": 500,
		"max_line_match_results": 10,
		"max_line_match_results_per_file": 3,
		"forward_to": [%s, %s]
	}`, forwardTo1, forwardTo2)

	req := &http.Request{
		Body:   io.NopCloser(strings.NewReader(body)),
		Header: make(http.Header),
	}
	req.Header.Set("Content-Type", "application/json")

	resp, err := searcher.Search(req)
	require.NoError(t, err)
	require.NotNil(t, resp)
	require.Len(t, resp.Result.Files, 2, "Expected NO deduplication of files from same repositories but different filenames")
	require.Equal(t, "file.go", resp.Result.Files[0].FileName)
	require.Equal(t, "file2.go", resp.Result.Files[1].FileName)
	require.Equal(t, resp.Result.Files[0].Repository, resp.Result.Files[1].Repository, "Files should be from same repository")
}
