package search

import (
	"errors"
)

func (sr *RawSearchRequest) getVersion() uint8 {
	if sr == nil || sr.Version == 0 {
		// Default to version 1 if not specified
		// This is to maintain backward compatibility with older versions
		return 1
	}
	return sr.Version
}

func (sr *RawSearchRequest) ToSearchRequest() (*SearchRequest, error) {
	switch sr.getVersion() {
	case 1:
		return convertV1ToV2(sr)
	case 2:
		return convertV2(sr)
	}

	return nil, errors.New("unsupported search request version")
}

func convertV2(sr *RawSearchRequest) (*SearchRequest, error) {
	// Already V2 format, just convert directly
	req := SearchRequest{
		TimeoutString:              sr.TimeoutString,
		NumContextLines:            sr.NumContextLines,
		MaxFileMatchWindow:         sr.MaxFileMatchWindow,
		MaxFileMatchResults:        sr.MaxFileMatchResults,
		MaxLineMatchWindow:         sr.MaxLineMatchWindow,
		MaxLineMatchResults:        sr.MaxLineMatchResults,
		MaxLineMatchResultsPerFile: sr.MaxLineMatchResultsPerFile,
		ForwardTo:                  sr.ForwardTo,
	}

	return &req, nil
}

func convertV1ToV2(sr *RawSearchRequest) (*SearchRequest, error) {
	v2 := &SearchRequest{
		TimeoutString:              sr.TimeoutStringV1,
		NumContextLines:            sr.Options.NumContextLines,
		MaxFileMatchWindow:         sr.Options.MaxFileMatchWindow,
		MaxFileMatchResults:        sr.Options.MaxFileMatchResults,
		MaxLineMatchWindow:         sr.Options.MaxLineMatchWindow,
		MaxLineMatchResults:        sr.Options.MaxLineMatchResults,
		MaxLineMatchResultsPerFile: sr.Options.MaxLineMatchResultsPerFile,
		ForwardTo:                  make([]Search, len(sr.ForwardToV1)),
	}

	if sr.Query == "" {
		return nil, errors.New("search query is empty")
	}

	// This constraint is for backward compatibility. Because traversal_ids is not support in V1,
	// we must ensure that repo_ids is not empty.
	for i, conn := range sr.ForwardToV1 {
		if len(conn.RepoIds) == 0 {
			return nil, errors.New("no repo IDs specified for forward-to connection")
		}

		v2.ForwardTo[i] = Search{
			Endpoint: conn.Endpoint,
			Query: Query{
				Data: map[string]interface{}{
					"and": map[string]interface{}{
						"children": []interface{}{
							map[string]interface{}{
								"repo_ids": conn.RepoIds,
							},
							map[string]interface{}{
								"query_string": map[string]interface{}{
									"query": sr.Query,
								},
							},
						},
					},
				},
			},
			Options: &EndpointOptions{
				Headers: sr.HeadersV1,
			},
		}

	}

	return v2, nil
}
