package search

import (
	"bytes"
	"encoding/json"
	"testing"

	"github.com/RoaringBitmap/roaring"
	proto "github.com/sourcegraph/zoekt/grpc/protos/zoekt/webserver/v1"
	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/require"
)

func TestQuery_UnmarshalJSON(t *testing.T) {
	tests := []struct {
		name     string
		jsonData string
		want     map[string]interface{}
		wantErr  bool
	}{
		{
			name:     "valid substring query",
			jsonData: `{"substring": {"pattern": "test", "case_sensitive": true}}`,
			want: map[string]interface{}{
				"substring": map[string]interface{}{
					"pattern":        "test",
					"case_sensitive": true,
				},
			},
		},
		{
			name:     "valid and query",
			jsonData: `{"and": {"children": [{"substring": {"pattern": "foo"}}, {"substring": {"pattern": "bar"}}]}}`,
			want: map[string]interface{}{
				"and": map[string]interface{}{
					"children": []interface{}{
						map[string]interface{}{
							"substring": map[string]interface{}{
								"pattern": "foo",
							},
						},
						map[string]interface{}{
							"substring": map[string]interface{}{
								"pattern": "bar",
							},
						},
					},
				},
			},
		},
		{
			name:     "invalid json",
			jsonData: `{invalid json}`,
			wantErr:  true,
		},
	}

	for _, tt := range tests {
		t.Run(tt.name, func(t *testing.T) {
			var q Query
			err := json.Unmarshal([]byte(tt.jsonData), &q)

			if tt.wantErr {
				assert.Error(t, err)
				return
			}

			require.NoError(t, err)
			assert.Equal(t, tt.want, q.Data)
		})
	}
}

func TestQuery_ToProto_EmptyQuery(t *testing.T) {
	q := Query{Data: map[string]interface{}{}}
	_, err := q.ToProto()
	assert.Error(t, err)
	assert.Contains(t, err.Error(), "empty query data")
}

func TestQuery_ToProto_MultipleTopLevelKeys(t *testing.T) {
	q := Query{Data: map[string]interface{}{
		"substring": map[string]interface{}{"pattern": "test"},
		"regexp":    map[string]interface{}{"regexp": "test.*"},
	}}
	_, err := q.ToProto()
	assert.Error(t, err)
	assert.Contains(t, err.Error(), "query must have exactly one top-level key")
}

func TestQuery_ToProto_UnsupportedQueryType(t *testing.T) {
	q := Query{Data: map[string]interface{}{
		"unsupported": map[string]interface{}{},
	}}
	_, err := q.ToProto()
	assert.Error(t, err)
	assert.Contains(t, err.Error(), "unsupported query type: unsupported")
}

func TestQuery_handleSubstring(t *testing.T) {
	tests := []struct {
		name    string
		data    interface{}
		want    *proto.Q
		wantErr bool
	}{
		{
			name: "valid substring with all fields",
			data: map[string]interface{}{
				"pattern":        "test",
				"case_sensitive": true,
				"file_name":      true,
				"content":        false,
			},
			want: &proto.Q{
				Query: &proto.Q_Substring{
					Substring: &proto.Substring{
						Pattern:       "test",
						CaseSensitive: true,
						FileName:      true,
						Content:       false,
					},
				},
			},
		},
		{
			name: "valid substring with only pattern",
			data: map[string]interface{}{
				"pattern": "simple",
			},
			want: &proto.Q{
				Query: &proto.Q_Substring{
					Substring: &proto.Substring{
						Pattern: "simple",
					},
				},
			},
		},
		{
			name:    "invalid data type",
			data:    "not an object",
			wantErr: true,
		},
		{
			name: "missing pattern",
			data: map[string]interface{}{
				"case_sensitive": true,
			},
			wantErr: true,
		},
		{
			name: "invalid pattern type",
			data: map[string]interface{}{
				"pattern": 123,
			},
			wantErr: true,
		},
	}

	for _, tt := range tests {
		t.Run(tt.name, func(t *testing.T) {
			q := &Query{}
			got, err := q.handleSubstring(tt.data)

			if tt.wantErr {
				assert.Error(t, err)
				return
			}

			require.NoError(t, err)
			assert.Equal(t, tt.want, got)
		})
	}
}

func TestQuery_handleRegexp(t *testing.T) {
	tests := []struct {
		name    string
		data    interface{}
		want    *proto.Q
		wantErr bool
	}{
		{
			name: "valid regexp with all fields",
			data: map[string]interface{}{
				"regexp":         "test.*",
				"case_sensitive": false,
				"file_name":      true,
				"content":        true,
			},
			want: &proto.Q{
				Query: &proto.Q_Regexp{
					Regexp: &proto.Regexp{
						Regexp:        "test.*",
						CaseSensitive: false,
						FileName:      true,
						Content:       true,
					},
				},
			},
		},
		{
			name: "valid regexp with only pattern",
			data: map[string]interface{}{
				"regexp": "^hello.*world$",
			},
			want: &proto.Q{
				Query: &proto.Q_Regexp{
					Regexp: &proto.Regexp{
						Regexp: "^hello.*world$",
					},
				},
			},
		},
		{
			name:    "invalid data type",
			data:    []string{"not", "an", "object"},
			wantErr: true,
		},
		{
			name: "missing regexp pattern",
			data: map[string]interface{}{
				"case_sensitive": true,
			},
			wantErr: true,
		},
		{
			name: "invalid regexp pattern type",
			data: map[string]interface{}{
				"regexp": 123.45,
			},
			wantErr: true,
		},
	}

	for _, tt := range tests {
		t.Run(tt.name, func(t *testing.T) {
			q := &Query{}
			got, err := q.handleRegexp(tt.data)

			if tt.wantErr {
				assert.Error(t, err)
				return
			}

			require.NoError(t, err)
			assert.Equal(t, tt.want, got)
		})
	}
}

func TestQuery_handleAnd(t *testing.T) {
	tests := []struct {
		name    string
		data    interface{}
		wantErr bool
	}{
		{
			name: "valid and query",
			data: map[string]interface{}{
				"children": []interface{}{
					map[string]interface{}{
						"substring": map[string]interface{}{
							"pattern": "foo",
						},
					},
					map[string]interface{}{
						"substring": map[string]interface{}{
							"pattern": "bar",
						},
					},
				},
			},
		},
		{
			name:    "invalid data type",
			data:    "not an object",
			wantErr: true,
		},
		{
			name: "missing children field",
			data: map[string]interface{}{
				"other_field": "value",
			},
			wantErr: true,
		},
		{
			name: "children not an array",
			data: map[string]interface{}{
				"children": "not an array",
			},
			wantErr: true,
		},
		{
			name: "invalid child query",
			data: map[string]interface{}{
				"children": []interface{}{
					map[string]interface{}{
						"invalid_query_type": map[string]interface{}{},
					},
				},
			},
			wantErr: true,
		},
	}

	for _, tt := range tests {
		t.Run(tt.name, func(t *testing.T) {
			q := &Query{}
			got, err := q.handleAnd(tt.data)

			if tt.wantErr {
				assert.Error(t, err)
				return
			}

			require.NoError(t, err)
			assert.NotNil(t, got)
			assert.NotNil(t, got.GetAnd())
			assert.Len(t, got.GetAnd().Children, 2)
		})
	}
}

func TestQuery_handleOr(t *testing.T) {
	tests := []struct {
		name    string
		data    interface{}
		wantErr bool
	}{
		{
			name: "valid or query",
			data: map[string]interface{}{
				"children": []interface{}{
					map[string]interface{}{
						"regexp": map[string]interface{}{
							"regexp": "test.*",
						},
					},
					map[string]interface{}{
						"substring": map[string]interface{}{
							"pattern": "example",
						},
					},
				},
			},
		},
		{
			name:    "invalid data type",
			data:    123,
			wantErr: true,
		},
		{
			name: "missing children field",
			data: map[string]interface{}{
				"some_field": "value",
			},
			wantErr: true,
		},
		{
			name: "children not an array",
			data: map[string]interface{}{
				"children": map[string]interface{}{
					"not": "an array",
				},
			},
			wantErr: true,
		},
	}

	for _, tt := range tests {
		t.Run(tt.name, func(t *testing.T) {
			q := &Query{}
			got, err := q.handleOr(tt.data)

			if tt.wantErr {
				assert.Error(t, err)
				return
			}

			require.NoError(t, err)
			assert.NotNil(t, got)
			assert.NotNil(t, got.GetOr())
			assert.Len(t, got.GetOr().Children, 2)
		})
	}
}

func TestQuery_handleNot(t *testing.T) {
	tests := []struct {
		name    string
		data    interface{}
		wantErr bool
	}{
		{
			name: "valid not query",
			data: map[string]interface{}{
				"child": map[string]interface{}{
					"substring": map[string]interface{}{
						"pattern": "exclude_this",
					},
				},
			},
		},
		{
			name:    "invalid data type",
			data:    []interface{}{},
			wantErr: true,
		},
		{
			name: "missing child field",
			data: map[string]interface{}{
				"other_field": "value",
			},
			wantErr: true,
		},
		{
			name: "invalid child query",
			data: map[string]interface{}{
				"child": map[string]interface{}{
					"unknown_type": map[string]interface{}{},
				},
			},
			wantErr: true,
		},
	}

	for _, tt := range tests {
		t.Run(tt.name, func(t *testing.T) {
			q := &Query{}
			got, err := q.handleNot(tt.data)

			if tt.wantErr {
				assert.Error(t, err)
				return
			}

			require.NoError(t, err)
			assert.NotNil(t, got)
			assert.NotNil(t, got.GetNot())
			assert.NotNil(t, got.GetNot().Child)
		})
	}
}

func TestQuery_handleSymbol(t *testing.T) {
	tests := []struct {
		name    string
		data    interface{}
		wantErr bool
	}{
		{
			name: "valid symbol query",
			data: map[string]interface{}{
				"expr": map[string]interface{}{
					"substring": map[string]interface{}{
						"pattern": "function_name",
					},
				},
			},
		},
		{
			name:    "invalid data type",
			data:    "not an object",
			wantErr: true,
		},
		{
			name: "missing expr field",
			data: map[string]interface{}{
				"other_field": "value",
			},
			wantErr: true,
		},
		{
			name: "invalid expr query",
			data: map[string]interface{}{
				"expr": map[string]interface{}{
					"invalid_query": map[string]interface{}{},
				},
			},
			wantErr: true,
		},
	}

	for _, tt := range tests {
		t.Run(tt.name, func(t *testing.T) {
			q := &Query{}
			got, err := q.handleSymbol(tt.data)

			if tt.wantErr {
				assert.Error(t, err)
				return
			}

			require.NoError(t, err)
			assert.NotNil(t, got)
			assert.NotNil(t, got.GetSymbol())
			assert.NotNil(t, got.GetSymbol().Expr)
		})
	}
}

func TestQuery_handleRepoIds(t *testing.T) {
	tests := []struct {
		name    string
		data    interface{}
		want    []uint32
		wantErr bool
	}{
		{
			name: "valid repo ids with float64",
			data: []interface{}{1.0, 2.0, 3.0},
			want: []uint32{1, 2, 3},
		},
		{
			name: "valid repo ids with int",
			data: []interface{}{1, 2, 3},
			want: []uint32{1, 2, 3},
		},
		{
			name: "mixed int and float64",
			data: []interface{}{1, 2.0, 3},
			want: []uint32{1, 2, 3},
		},
		{
			name:    "invalid data type",
			data:    "not an array",
			wantErr: true,
		},
		{
			name:    "invalid element type",
			data:    []interface{}{1, "string", 3},
			wantErr: true,
		},
		{
			name: "empty array",
			data: []interface{}{},
			want: []uint32{},
		},
	}

	for _, tt := range tests {
		t.Run(tt.name, func(t *testing.T) {
			q := &Query{}
			got, err := q.handleRepoIds(tt.data)

			if tt.wantErr {
				assert.Error(t, err)
				return
			}

			require.NoError(t, err)
			assert.NotNil(t, got)
			assert.NotNil(t, got.GetRepoIds())

			// Verify the bitmap contains the expected repo IDs
			bitmap := roaring.NewBitmap()
			_, err = bitmap.ReadFrom(bytes.NewReader(got.GetRepoIds().Repos))
			require.NoError(t, err)

			expectedBitmap := roaring.NewBitmap()
			for _, id := range tt.want {
				expectedBitmap.Add(id)
			}

			assert.True(t, bitmap.Equals(expectedBitmap), "bitmaps should be equal")
		})
	}
}

func TestQuery_handleMeta(t *testing.T) {
	tests := []struct {
		name    string
		data    interface{}
		want    *proto.Q
		wantErr bool
	}{
		{
			name: "valid meta query",
			data: map[string]interface{}{
				"key":   "foo",
				"value": "bar",
			},
			want: &proto.Q{
				Query: &proto.Q_Meta{
					Meta: &proto.Meta{
						Key:   "foo",
						Value: "bar",
					},
				},
			},
		},
		{
			name:    "invalid data type",
			data:    "not an object",
			wantErr: true,
		},
		{
			name: "missing key field",
			data: map[string]interface{}{
				"value": "bar",
			},
			wantErr: true,
		},
		{
			name: "missing value field",
			data: map[string]interface{}{
				"key": "foo",
			},
			wantErr: true,
		},
		{
			name: "key not a string",
			data: map[string]interface{}{
				"key":   123,
				"value": "bar",
			},
			wantErr: true,
		},
		{
			name: "value not a string",
			data: map[string]interface{}{
				"key":   "foo",
				"value": 456,
			},
			wantErr: true,
		},
	}

	for _, tt := range tests {
		t.Run(tt.name, func(t *testing.T) {
			q := &Query{}
			got, err := q.handleMeta(tt.data)

			if tt.wantErr {
				assert.Error(t, err)
				return
			}

			require.NoError(t, err)
			assert.Equal(t, tt.want, got)
		})
	}
}

func TestDataToQuery(t *testing.T) {
	tests := []struct {
		name    string
		data    interface{}
		wantErr bool
	}{
		{
			name: "valid query data",
			data: map[string]interface{}{
				"substring": map[string]interface{}{
					"pattern": "test",
				},
			},
		},
		{
			name:    "unmarshalable data",
			data:    make(chan int), // channels can't be marshaled to JSON
			wantErr: true,
		},
	}

	for _, tt := range tests {
		t.Run(tt.name, func(t *testing.T) {
			got, err := dataToQuery(tt.data)

			if tt.wantErr {
				assert.Error(t, err)
				return
			}

			require.NoError(t, err)
			assert.NotNil(t, got)
			assert.NotNil(t, got.Data)
		})
	}
}

func TestQuery_ToProto_Integration(t *testing.T) {
	tests := []struct {
		name     string
		jsonData string
		wantErr  bool
	}{
		{
			name:     "complex nested query",
			jsonData: `{"and": {"children": [{"substring": {"pattern": "foo"}}, {"or": {"children": [{"regexp": {"regexp": "bar.*"}}, {"not": {"child": {"substring": {"pattern": "baz"}}}}]}}]}}`,
		},
		{
			name:     "symbol with regexp",
			jsonData: `{"symbol": {"expr": {"regexp": {"regexp": "func.*", "case_sensitive": true}}}}`,
		},
		{
			name:     "repo ids query",
			jsonData: `{"repo_ids": [1, 2, 3, 100, 1000]}`,
		},
	}

	for _, tt := range tests {
		t.Run(tt.name, func(t *testing.T) {
			var q Query
			err := json.Unmarshal([]byte(tt.jsonData), &q)
			require.NoError(t, err)

			protoQ, err := q.ToProto()
			if tt.wantErr {
				assert.Error(t, err)
				return
			}

			require.NoError(t, err)
			assert.NotNil(t, protoQ)
		})
	}
}

func TestQuery_ToProto_ContextKeyAllowed(t *testing.T) {
	q := Query{Data: map[string]interface{}{
		"substring": map[string]interface{}{"pattern": "test"},
		"_context":  map[string]interface{}{"name": "something", "foo": "bar"},
	}}
	protoQ, err := q.ToProto()
	assert.NoError(t, err)
	assert.NotNil(t, protoQ)
}

func TestQuery_ToProto_MultipleTopLevelKeys_WithContext(t *testing.T) {
	q := Query{Data: map[string]interface{}{
		"substring": map[string]interface{}{"pattern": "test"},
		"regexp":    map[string]interface{}{"regexp": "test.*"},
		"_context":  map[string]interface{}{"name": "something", "foo": "bar"},
	}}
	_, err := q.ToProto()
	assert.Error(t, err)
	assert.Contains(t, err.Error(), "query must have exactly one top-level key")
}
