package search

import (
	"bytes"
	"encoding/json"
	"errors"
	"fmt"
	"strings"

	"github.com/RoaringBitmap/roaring"
	proto "github.com/sourcegraph/zoekt/grpc/protos/zoekt/webserver/v1"
	zquery "github.com/sourcegraph/zoekt/query"
)

type Query struct {
	Data map[string]interface{} `json:"-"`
}

func (q *Query) UnmarshalJSON(data []byte) error {
	return json.Unmarshal(data, &q.Data)
}

func (q *Query) ToProto() (*proto.Q, error) {
	if len(q.Data) == 0 {
		return nil, fmt.Errorf("empty query data")
	}

	// Only allow one top-level key, or two if one is _context
	keys := make([]string, 0, len(q.Data))
	for k := range q.Data {
		keys = append(keys, k)
	}

	mainKeys := make([]string, 0, len(keys))
	for _, k := range keys {
		if k != "_context" {
			mainKeys = append(mainKeys, k)
		}
	}

	if len(mainKeys) != 1 {
		return nil, fmt.Errorf("query must have exactly one top-level key (and optional _context), got %d: %s", len(q.Data), strings.Join(keys, ", "))
	}

	for queryType, queryData := range q.Data {
		if queryType == "_context" {
			continue // skip _context
		}
		switch queryType {
		case "and":
			return q.handleAnd(queryData)
		case "or":
			return q.handleOr(queryData)
		case "not":
			return q.handleNot(queryData)
		case "symbol":
			return q.handleSymbol(queryData)
		case "substring":
			return q.handleSubstring(queryData)
		case "regexp":
			return q.handleRegexp(queryData)
		case "repo_ids":
			return q.handleRepoIds(queryData)
		case "meta":
			return q.handleMeta(queryData)
		case "query_string":
			return q.handleQueryString(queryData)
		default:
			return nil, fmt.Errorf("unsupported query type: %s", queryType)
		}
	}

	return nil, fmt.Errorf("no query type found")
}

func (q *Query) handleSubstring(data interface{}) (*proto.Q, error) {
	// Convert interface{} to map[string]interface{}
	substringData, ok := data.(map[string]interface{})
	if !ok {
		return nil, fmt.Errorf("substring data must be an object")
	}

	substring := &proto.Substring{}

	// Extract pattern (required)
	if pattern, exists := substringData["pattern"]; exists {
		if patternStr, ok := pattern.(string); ok {
			substring.Pattern = patternStr
		} else {
			return nil, fmt.Errorf("substring pattern must be a string")
		}
	} else {
		return nil, fmt.Errorf("substring pattern is required")
	}

	// Extract optional boolean fields
	if caseSensitive, exists := substringData["case_sensitive"]; exists {
		if caseSensitiveBool, ok := caseSensitive.(bool); ok {
			substring.CaseSensitive = caseSensitiveBool
		}
	}

	if fileName, exists := substringData["file_name"]; exists {
		if fileNameBool, ok := fileName.(bool); ok {
			substring.FileName = fileNameBool
		}
	}

	if content, exists := substringData["content"]; exists {
		if contentBool, ok := content.(bool); ok {
			substring.Content = contentBool
		}
	}

	return &proto.Q{
		Query: &proto.Q_Substring{
			Substring: substring,
		},
	}, nil
}

func (q *Query) handleRegexp(data interface{}) (*proto.Q, error) {
	// Convert interface{} to map[string]interface{}
	regexpData, ok := data.(map[string]interface{})
	if !ok {
		return nil, fmt.Errorf("regexp data must be an object")
	}

	regexp := &proto.Regexp{}

	// Extract regexp pattern (required)
	if pattern, exists := regexpData["regexp"]; exists {
		if patternStr, ok := pattern.(string); ok {
			regexp.Regexp = patternStr
		} else {
			return nil, fmt.Errorf("regexp pattern must be a string")
		}
	} else {
		return nil, fmt.Errorf("regexp pattern is required")
	}

	// Extract optional boolean fields
	if fileName, exists := regexpData["file_name"]; exists {
		if fileNameBool, ok := fileName.(bool); ok {
			regexp.FileName = fileNameBool
		}
	}

	if content, exists := regexpData["content"]; exists {
		if contentBool, ok := content.(bool); ok {
			regexp.Content = contentBool
		}
	}

	if caseSensitive, exists := regexpData["case_sensitive"]; exists {
		if caseSensitiveBool, ok := caseSensitive.(bool); ok {
			regexp.CaseSensitive = caseSensitiveBool
		}
	}

	return &proto.Q{
		Query: &proto.Q_Regexp{
			Regexp: regexp,
		},
	}, nil
}

// Helper function to convert interface{} back to Query
func dataToQuery(data interface{}) (*Query, error) {
	// Convert back to JSON bytes, then unmarshal into Query
	jsonBytes, err := json.Marshal(data)
	if err != nil {
		return nil, fmt.Errorf("failed to marshal query data: %w", err)
	}

	var query Query
	if err := json.Unmarshal(jsonBytes, &query); err != nil {
		return nil, fmt.Errorf("failed to unmarshal query data: %w", err)
	}

	return &query, nil
}

func (q *Query) handleAnd(data interface{}) (*proto.Q, error) {
	andData, ok := data.(map[string]interface{})
	if !ok {
		return nil, fmt.Errorf("and data must be an object")
	}

	childrenData, exists := andData["children"]
	if !exists {
		return nil, fmt.Errorf("and query requires 'children' field")
	}

	childrenArray, ok := childrenData.([]interface{})
	if !ok {
		return nil, fmt.Errorf("and children must be an array of objects")
	}

	children := make([]*proto.Q, 0, len(childrenArray))
	for i, childData := range childrenArray {
		childQuery, err := dataToQuery(childData)
		if err != nil {
			return nil, fmt.Errorf("failed to parse and child %d: %w", i, err)
		}

		childProto, err := childQuery.ToProto()
		if err != nil {
			return nil, fmt.Errorf("failed to convert and child %d to proto: %w", i, err)
		}

		children = append(children, childProto)
	}

	return &proto.Q{
		Query: &proto.Q_And{
			And: &proto.And{
				Children: children,
			},
		},
	}, nil
}

func (q *Query) handleOr(data interface{}) (*proto.Q, error) {
	orData, ok := data.(map[string]interface{})
	if !ok {
		return nil, fmt.Errorf("or data must be an object")
	}

	childrenData, exists := orData["children"]
	if !exists {
		return nil, fmt.Errorf("or query requires 'children' field")
	}

	childrenArray, ok := childrenData.([]interface{})
	if !ok {
		return nil, fmt.Errorf("or children must be an array")
	}

	children := make([]*proto.Q, 0, len(childrenArray))
	for i, childData := range childrenArray {
		childQuery, err := dataToQuery(childData)
		if err != nil {
			return nil, fmt.Errorf("failed to parse or child %d: %w", i, err)
		}

		childProto, err := childQuery.ToProto()
		if err != nil {
			return nil, fmt.Errorf("failed to convert or child %d to proto: %w", i, err)
		}

		children = append(children, childProto)
	}

	return &proto.Q{
		Query: &proto.Q_Or{
			Or: &proto.Or{
				Children: children,
			},
		},
	}, nil
}

func (q *Query) handleNot(data interface{}) (*proto.Q, error) {
	notData, ok := data.(map[string]interface{})
	if !ok {
		return nil, fmt.Errorf("not data must be an object")
	}

	childData, exists := notData["child"]
	if !exists {
		return nil, fmt.Errorf("not query requires 'child' field")
	}

	childQuery, err := dataToQuery(childData)
	if err != nil {
		return nil, fmt.Errorf("failed to parse not child: %w", err)
	}

	childProto, err := childQuery.ToProto()
	if err != nil {
		return nil, fmt.Errorf("failed to convert not child to proto: %w", err)
	}

	return &proto.Q{
		Query: &proto.Q_Not{
			Not: &proto.Not{
				Child: childProto,
			},
		},
	}, nil
}

func (q *Query) handleSymbol(data interface{}) (*proto.Q, error) {
	symbolData, ok := data.(map[string]interface{})
	if !ok {
		return nil, fmt.Errorf("symbol data must be an object")
	}

	exprData, exists := symbolData["expr"]
	if !exists {
		return nil, fmt.Errorf("symbol query requires 'expr' field")
	}

	exprQuery, err := dataToQuery(exprData)
	if err != nil {
		return nil, fmt.Errorf("failed to parse symbol expr: %w", err)
	}

	exprProto, err := exprQuery.ToProto()
	if err != nil {
		return nil, fmt.Errorf("failed to convert symbol expr to proto: %w", err)
	}

	return &proto.Q{
		Query: &proto.Q_Symbol{
			Symbol: &proto.Symbol{
				Expr: exprProto,
			},
		},
	}, nil
}

func (q *Query) handleRepoIds(data interface{}) (*proto.Q, error) {
	repoIdsArray, ok := data.([]interface{})
	if !ok {
		return nil, fmt.Errorf("repo_ids must be an array")
	}

	// Convert to uint32 slice
	var repoIds []uint32
	for i, idInterface := range repoIdsArray {
		switch id := idInterface.(type) {
		case float64:
			repoIds = append(repoIds, uint32(id))
		case int:
			repoIds = append(repoIds, uint32(id)) // #nosec G115
		default:
			return nil, fmt.Errorf("repo_ids[%d] must be a number, got %T", i, id)
		}
	}

	// Create roaring bitmap and add all repo IDs
	bitmap := roaring.NewBitmap()
	for _, id := range repoIds {
		bitmap.Add(id)
	}

	// Serialize the bitmap
	var buf bytes.Buffer
	if _, err := bitmap.WriteTo(&buf); err != nil {
		return nil, fmt.Errorf("failed to serialize repo IDs: %w", err)
	}

	return &proto.Q{
		Query: &proto.Q_RepoIds{
			RepoIds: &proto.RepoIds{
				Repos: buf.Bytes(),
			},
		},
	}, nil
}

func (q *Query) handleMeta(data interface{}) (*proto.Q, error) {
	metaData, ok := data.(map[string]interface{})
	if !ok {
		return nil, fmt.Errorf("meta data must be an object")
	}

	meta := &proto.Meta{}

	// Extract key
	if key, exists := metaData["key"]; exists {
		if keyStr, ok := key.(string); ok {
			meta.Key = keyStr
		} else {
			return nil, fmt.Errorf("meta key must be a string")
		}
	} else {
		return nil, fmt.Errorf("meta key is required")
	}

	// Extract value
	if value, exists := metaData["value"]; exists {
		if valueStr, ok := value.(string); ok {
			meta.Value = valueStr
		} else {
			return nil, fmt.Errorf("meta value must be a string")
		}
	} else {
		return nil, fmt.Errorf("meta value is required")
	}

	return &proto.Q{
		Query: &proto.Q_Meta{
			Meta: meta,
		},
	}, nil
}

func (q *Query) handleQueryString(data interface{}) (*proto.Q, error) {
	queryData, ok := data.(map[string]interface{})
	if !ok {
		return nil, fmt.Errorf("query_string data must be an object")
	}

	pattern, exists := queryData["query"]
	if !exists {
		return nil, fmt.Errorf("query_string requires 'query' field")
	}
	queryString, ok := pattern.(string)
	if !ok {
		return nil, fmt.Errorf("query_string query must be a string")
	}

	textQuery, err := zquery.Parse(queryString)
	if err != nil {
		return nil, errors.New("failed to parse search query: " + err.Error())
	}

	return zquery.QToProto(textQuery), nil
}
