package webserver

import (
	"bytes"
	"encoding/json"
	"io"
	"net/http"
	"net/http/httptest"
	"testing"
	"time"

	"github.com/sourcegraph/zoekt"
	"github.com/stretchr/testify/require"
	"gitlab.com/gitlab-org/gitlab-zoekt-indexer/internal/search"
	"google.golang.org/grpc"
)

// TestSearchV2 ensures the /webserver/api/v2/search endpoint returns the correct merged results.
func TestSearchV2(t *testing.T) {
	t.Parallel()

	// Create gRPC mock server 1
	files1 := search.ToProtoFileMatches([]zoekt.FileMatch{
		{FileName: "file1.go", Repository: "repo1", Score: 0.6, Checksum: []byte{0x1, 0x2, 0x3}},
	})
	conn1, endpoint1 := search.StartBufconnServer(t, files1, nil)
	defer conn1.Close()

	// Create gRPC mock server 2
	files2 := search.ToProtoFileMatches([]zoekt.FileMatch{
		{FileName: "file2.go", Repository: "repo2", Score: 0.9, Checksum: []byte{0x4, 0x5, 0x6}},
	})
	conn2, endpoint2 := search.StartBufconnServer(t, files2, nil)
	defer conn2.Close()

	// Inject gRPC connections into the search client
	searcher := &search.Searcher{
		Client:    &http.Client{Timeout: 2 * time.Second},
		GrpcConns: map[string]*grpc.ClientConn{search.GetHostFromEndpoint(endpoint1): conn1, search.GetHostFromEndpoint(endpoint2): conn2},
	}

	// Inject the custom searcher into your mux via a test-only wrapper
	handler := http.NewServeMux()
	handler.HandleFunc("/webserver/api/v2/search", func(w http.ResponseWriter, r *http.Request) {
		result, err := searcher.Search(r)
		if err != nil {
			http.Error(w, err.Error(), http.StatusInternalServerError)
			return
		}
		w.Header().Set("Content-Type", "application/json")
		_ = json.NewEncoder(w).Encode(result)
	})

	// Prepare request payload
	payload := map[string]any{
		"Q": "test",
		"Opts": map[string]any{
			"TotalMaxMatchCount": 10,
			"NumContextLines":    2,
		},
		"Timeout": "1s",
		"ForwardTo": []map[string]any{
			{"Endpoint": endpoint1, "RepoIds": []int{1, 2}},
			{"Endpoint": endpoint2, "RepoIds": []int{3}},
		},
		"grpc": true,
	}
	body, err := json.Marshal(payload)
	require.NoError(t, err)

	req := httptest.NewRequest(http.MethodPost, "/webserver/api/v2/search", bytes.NewReader(body))
	req.Header.Set("Content-Type", "application/json")
	rec := httptest.NewRecorder()

	handler.ServeHTTP(rec, req)

	resp := rec.Result()
	defer resp.Body.Close()

	require.Equal(t, http.StatusOK, resp.StatusCode)

	var result search.SearchResult
	err = json.NewDecoder(resp.Body).Decode(&result)
	require.NoError(t, err)

	require.Len(t, result.Result.Files, 2)
	require.Equal(t, "file2.go", result.Result.Files[0].FileName)
	require.Equal(t, "file1.go", result.Result.Files[1].FileName)
}

func TestSearchV2_AllGrpcFail(t *testing.T) {
	t.Parallel()

	// Setup a mock gRPC server that always fails
	conn, endpoint := search.StartFailingBufconnServer(t, "Internal Server Error")
	defer conn.Close()

	// Create the failing searcher with injected connection
	searcher := &search.Searcher{
		Client:    &http.Client{Timeout: 2 * time.Second},
		GrpcConns: map[string]*grpc.ClientConn{search.GetHostFromEndpoint(endpoint): conn},
	}

	// Setup V2 handler
	handler := http.NewServeMux()
	handler.HandleFunc("/webserver/api/v2/search", func(w http.ResponseWriter, r *http.Request) {
		result, err := searcher.Search(r)
		if err != nil {
			http.Error(w, err.Error(), http.StatusInternalServerError)
			return
		}
		w.Header().Set("Content-Type", "application/json")
		_ = json.NewEncoder(w).Encode(result)
	})

	// Build request payload with only the failing endpoint
	payload := map[string]any{
		"Q": "test",
		"Opts": map[string]any{
			"TotalMaxMatchCount": 10,
			"NumContextLines":    2,
		},
		"Timeout": "1s",
		"ForwardTo": []map[string]any{
			{"Endpoint": endpoint, "RepoIds": []int{1, 2}},
		},
		"grpc": true,
	}
	body, err := json.Marshal(payload)
	require.NoError(t, err)

	req := httptest.NewRequest("POST", "/webserver/api/v2/search", bytes.NewReader(body))
	req.Header.Set("Content-Type", "application/json")
	rec := httptest.NewRecorder()

	handler.ServeHTTP(rec, req)

	resp := rec.Result()
	defer resp.Body.Close()

	// Check that it's a 500 response and includes expected error info
	require.Equal(t, http.StatusInternalServerError, resp.StatusCode)

	bodyBytes, err := io.ReadAll(resp.Body)
	require.NoError(t, err)
	require.Contains(t, string(bodyBytes), "all searches failed")
}
