package webserver

import (
	"flag"
	"io"
	"os"
	"path/filepath"
	"testing"

	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/require"
)

// createTestSecretFile creates a temporary secret file for testing
func createTestSecretFile(t *testing.T) string {
	secretFile := filepath.Join(t.TempDir(), "secret")
	err := os.WriteFile(secretFile, []byte("test-secret-key"), 0600)
	require.NoError(t, err)
	return secretFile
}

func TestParseFlagsWithIndexDir(t *testing.T) {
	// Create a temporary directory for testing
	tempDir := t.TempDir()
	indexDir := filepath.Join(tempDir, "index")
	secretFile := createTestSecretFile(t)

	// Save original command line arguments and restore after test
	origArgs := os.Args
	origFlagCommandLine := flag.CommandLine
	defer func() {
		os.Args = origArgs
		flag.CommandLine = origFlagCommandLine
	}()

	// Reset flags for the test
	flag.CommandLine = flag.NewFlagSet(os.Args[0], flag.ContinueOnError)
	flag.CommandLine.SetOutput(io.Discard) // Suppress flag output during test

	// Setup test arguments
	os.Args = []string{
		"cmd",
		"-index_dir", indexDir,
		"-listen", ":8080",
		"-secret_path", secretFile,
	}

	// Parse flags
	opts, err := ParseFlags()
	require.NoError(t, err)

	// Verify the index_dir flag is processed correctly
	assert.Equal(t, indexDir, opts.BaseDir)
	assert.Equal(t, ":8080", opts.Listen)
}

func TestParseFlagsWithLegacyIndexFlag(t *testing.T) {
	// Create a temporary directory for testing
	tempDir := t.TempDir()
	indexDir := filepath.Join(tempDir, "index")
	secretFile := createTestSecretFile(t)

	// Save original command line arguments and restore after test
	origArgs := os.Args
	origFlagCommandLine := flag.CommandLine
	defer func() {
		os.Args = origArgs
		flag.CommandLine = origFlagCommandLine
	}()

	// Reset flags for the test
	flag.CommandLine = flag.NewFlagSet(os.Args[0], flag.ContinueOnError)
	flag.CommandLine.SetOutput(io.Discard) // Suppress flag output during test

	// Suppress stderr output for the test
	devNull, _ := os.Open(os.DevNull)
	defer devNull.Close()
	os.Stderr = devNull

	// Setup test arguments with legacy flag
	os.Args = []string{
		"cmd",
		"-index", indexDir,
		"-listen", ":8080",
		"-secret_path", secretFile,
	}

	// Parse flags
	opts, err := ParseFlags()
	require.NoError(t, err)

	// Verify the legacy flag is processed correctly
	assert.Equal(t, indexDir, opts.BaseDir)
	// Note: We can't easily test the warning, as it's sent via slog
}

func TestParseFlagsWithBothIndexFlags(t *testing.T) {
	// Create a temporary directory for testing
	tempDir := t.TempDir()
	legacyIndexDir := filepath.Join(tempDir, "legacy-index")
	newIndexDir := filepath.Join(tempDir, "new-index")
	secretFile := createTestSecretFile(t)

	// Save original command line arguments and restore after test
	origArgs := os.Args
	origFlagCommandLine := flag.CommandLine
	defer func() {
		os.Args = origArgs
		flag.CommandLine = origFlagCommandLine
	}()

	// Reset flags for the test
	flag.CommandLine = flag.NewFlagSet(os.Args[0], flag.ContinueOnError)
	flag.CommandLine.SetOutput(io.Discard) // Suppress flag output during test

	// Redirect standard error to discard warnings
	oldStderr := os.Stderr
	devNull, _ := os.Open(os.DevNull)
	os.Stderr = devNull
	defer devNull.Close()

	// Setup test arguments with both flags
	os.Args = []string{
		"cmd",
		"-index", legacyIndexDir,
		"-index_dir", newIndexDir,
		"-listen", ":8080",
		"-secret_path", secretFile,
	}

	// Parse flags
	opts, err := ParseFlags()

	// Restore stderr
	os.Stderr = oldStderr

	require.NoError(t, err)

	// Verify the new flag takes precedence
	assert.Equal(t, newIndexDir, opts.BaseDir)
}

func TestParseFlagsRequiresSecretPath(t *testing.T) {
	// Create a temporary directory for testing
	tempDir := t.TempDir()
	indexDir := filepath.Join(tempDir, "index")

	// Save original command line arguments and restore after test
	origArgs := os.Args
	origFlagCommandLine := flag.CommandLine
	defer func() {
		os.Args = origArgs
		flag.CommandLine = origFlagCommandLine
	}()

	// Reset flags for the test
	flag.CommandLine = flag.NewFlagSet(os.Args[0], flag.ContinueOnError)
	flag.CommandLine.SetOutput(io.Discard) // Suppress flag output during test

	// Setup test arguments WITHOUT secret_path
	os.Args = []string{
		"cmd",
		"-index_dir", indexDir,
		"-listen", ":8080",
	}

	// Parse flags - should fail
	opts, err := ParseFlags()
	require.Error(t, err)
	require.Nil(t, opts)
	require.Contains(t, err.Error(), "secret_path is required")
}
