package auth

import (
	"bytes"
	"log/slog"
	"net/http"
	"net/http/httptest"
	"testing"
	"time"

	"github.com/stretchr/testify/require"
	"gitlab.com/gitlab-org/gitlab-zoekt-indexer/internal/authentication"
)

func createTestAuth() authentication.Auth {
	return authentication.NewAuth("test-issuer", time.Hour, []byte("test-secret"))
}

func createTestConfig(enabled bool) *Config {
	var logBuffer bytes.Buffer
	logger := slog.New(slog.NewJSONHandler(&logBuffer, &slog.HandlerOptions{Level: slog.LevelDebug}))

	auth := createTestAuth()
	return &Config{
		Auth:          &auth,
		Logger:        logger,
		ExcludedPaths: []string{"/health", "/metrics"},
		Enabled:       enabled,
	}
}

func TestMiddleware_Disabled(t *testing.T) {
	config := createTestConfig(false)

	handler := config.Middleware()(http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
		w.WriteHeader(http.StatusOK)
		w.Write([]byte("success"))
	}))

	req := httptest.NewRequest("GET", "/test", nil)
	w := httptest.NewRecorder()

	handler.ServeHTTP(w, req)

	require.Equal(t, http.StatusOK, w.Code)
	require.Equal(t, "success", w.Body.String())
}

func TestMiddleware_ExcludedPaths(t *testing.T) {
	config := createTestConfig(true)

	handler := config.Middleware()(http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
		w.WriteHeader(http.StatusOK)
		w.Write([]byte("success"))
	}))

	// Test excluded paths
	excludedPaths := []string{"/health", "/metrics"}
	for _, path := range excludedPaths {
		req := httptest.NewRequest("GET", path, nil)
		w := httptest.NewRecorder()

		handler.ServeHTTP(w, req)

		require.Equal(t, http.StatusOK, w.Code, "Path %s should be excluded", path)
		require.Equal(t, "success", w.Body.String())
	}
}

func TestMiddleware_MissingAuthHeader(t *testing.T) {
	config := createTestConfig(true)

	var logBuffer bytes.Buffer
	config.Logger = slog.New(slog.NewJSONHandler(&logBuffer, &slog.HandlerOptions{Level: slog.LevelDebug}))

	handler := config.Middleware()(http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
		w.WriteHeader(http.StatusOK)
		w.Write([]byte("success"))
	}))

	req := httptest.NewRequest("GET", "/test", nil)
	w := httptest.NewRecorder()

	handler.ServeHTTP(w, req)

	require.Equal(t, http.StatusUnauthorized, w.Code)
	require.Contains(t, w.Body.String(), "Unauthorized: missing "+authentication.GitlabZoektAPIRequestHeader+" header")

	// Check that warning was logged
	logOutput := logBuffer.String()
	require.Contains(t, logOutput, "missing "+authentication.GitlabZoektAPIRequestHeader+" header")
}

func TestMiddleware_ValidToken(t *testing.T) {
	config := createTestConfig(true)

	var logBuffer bytes.Buffer
	config.Logger = slog.New(slog.NewJSONHandler(&logBuffer, &slog.HandlerOptions{Level: slog.LevelDebug}))

	handler := config.Middleware()(http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
		w.WriteHeader(http.StatusOK)
		w.Write([]byte("success"))
	}))

	// Generate a valid token
	token, err := config.Auth.GenerateJWT()
	require.NoError(t, err)

	req := httptest.NewRequest("GET", "/test", nil)
	req.Header.Set(authentication.GitlabZoektAPIRequestHeader, "Bearer "+token)
	w := httptest.NewRecorder()

	handler.ServeHTTP(w, req)

	require.Equal(t, http.StatusOK, w.Code)
	require.Equal(t, "success", w.Body.String())

	// Should not contain any warning logs
	logOutput := logBuffer.String()
	require.NotContains(t, logOutput, "missing "+authentication.GitlabZoektAPIRequestHeader+" header")
	require.NotContains(t, logOutput, "Invalid JWT token")
}

func TestMiddleware_InvalidToken(t *testing.T) {
	config := createTestConfig(true)

	var logBuffer bytes.Buffer
	config.Logger = slog.New(slog.NewJSONHandler(&logBuffer, &slog.HandlerOptions{Level: slog.LevelDebug}))

	handler := config.Middleware()(http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
		w.WriteHeader(http.StatusOK)
		w.Write([]byte("success"))
	}))

	req := httptest.NewRequest("GET", "/test", nil)
	req.Header.Set(authentication.GitlabZoektAPIRequestHeader, "Bearer invalid.token.here")
	w := httptest.NewRecorder()

	handler.ServeHTTP(w, req)

	require.Equal(t, http.StatusUnauthorized, w.Code)
	require.Contains(t, w.Body.String(), "Unauthorized: invalid JWT token")

	// Check that warning was logged
	logOutput := logBuffer.String()
	require.Contains(t, logOutput, "Invalid JWT token")
}

func TestMiddleware_NoBearerPrefix(t *testing.T) {
	config := createTestConfig(true)

	var logBuffer bytes.Buffer
	config.Logger = slog.New(slog.NewJSONHandler(&logBuffer, &slog.HandlerOptions{Level: slog.LevelDebug}))

	handler := config.Middleware()(http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
		w.WriteHeader(http.StatusOK)
		w.Write([]byte("success"))
	}))

	// Generate a valid token but don't use Bearer prefix
	token, err := config.Auth.GenerateJWT()
	require.NoError(t, err)

	req := httptest.NewRequest("GET", "/test", nil)
	req.Header.Set(authentication.GitlabZoektAPIRequestHeader, token)
	w := httptest.NewRecorder()

	handler.ServeHTTP(w, req)

	require.Equal(t, http.StatusUnauthorized, w.Code)
	require.Contains(t, w.Body.String(), "Unauthorized: invalid JWT token")

	// Check that warning was logged
	logOutput := logBuffer.String()
	require.Contains(t, logOutput, "Invalid JWT token")
	require.Contains(t, logOutput, "authorization header must start with 'Bearer '")
}

func TestIsExcludedPath(t *testing.T) {
	config := &Config{
		ExcludedPaths: []string{"/health", "/metrics", "/api/v1/public"},
	}

	testCases := []struct {
		path     string
		excluded bool
	}{
		{"/", false},
		{"/health", true},
		{"/metrics", true},
		{"/api/v1/public", true},
		{"/api/v1/public/test", true}, // prefix match
		{"/test", false},
		{"/api/v1/private", false},
		{"/healthz", false}, // partial match should not work
	}

	for _, tc := range testCases {
		result := config.isExcludedPath(tc.path)
		require.Equal(t, tc.excluded, result, "Path %s exclusion result mismatch", tc.path)
	}
}
