// Package auth_middleware provides HTTP middleware for JWT authentication verification.
// It validates GitlabZoektAPIRequestHeader containing Bearer tokens and logs warnings
// when tokens are missing or invalid.
package auth

import (
	"log/slog"
	"net/http"
	"strings"

	"gitlab.com/gitlab-org/gitlab-zoekt-indexer/internal/authentication"
)

// Config holds the configuration for the JWT authentication middleware
type Config struct {
	Auth          *authentication.Auth
	Logger        *slog.Logger
	ExcludedPaths []string // Paths that don't require authentication (e.g., /health, /metrics)
	Enabled       bool     // Whether JWT verification is enabled
}

// Middleware returns an HTTP middleware that verifies JWT tokens in GitlabZoektAPIRequestHeader
func (c *Config) Middleware() func(next http.Handler) http.Handler {
	return func(next http.Handler) http.Handler {
		return http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
			// If JWT verification is not enabled, just pass through
			if !c.Enabled {
				next.ServeHTTP(w, r)
				return
			}

			// Check if the path is excluded from authentication
			if c.isExcludedPath(r.URL.Path) {
				next.ServeHTTP(w, r)
				return
			}

			authHeader := r.Header.Get(authentication.GitlabZoektAPIRequestHeader)

			// If no GitlabZoektAPIRequestHeader, return 401 Unauthorized
			if authHeader == "" {
				c.Logger.Warn("missing "+authentication.GitlabZoektAPIRequestHeader+" header", "method", r.Method, "path", r.URL.Path, "remote_addr", r.RemoteAddr)
				http.Error(w, "Unauthorized: missing "+authentication.GitlabZoektAPIRequestHeader+" header", http.StatusUnauthorized)
				return
			}

			// Verify the Bearer token
			_, err := c.Auth.VerifyBearerToken(authHeader)
			if err != nil {
				c.Logger.Warn("Invalid JWT token", "method", r.Method, "path", r.URL.Path, "remote_addr", r.RemoteAddr, "error", err.Error())
				c.Logger.Debug("Invalid JWT token", "token", authHeader)
				http.Error(w, "Unauthorized: invalid JWT token", http.StatusUnauthorized)
				return
			}

			// Token is valid, continue processing
			next.ServeHTTP(w, r)
		})
	}
}

// isExcludedPath checks if the given path should be excluded from JWT verification
func (c *Config) isExcludedPath(path string) bool {
	for _, excludedPath := range c.ExcludedPaths {
		// Exact match
		if path == excludedPath {
			return true
		}
		// Prefix match only if the excluded path ends with "/" or if the path starts with excludedPath + "/"
		// Prefix match for paths
		if strings.HasSuffix(excludedPath, "/") && strings.HasPrefix(path, excludedPath) {
			return true
		} else if strings.HasPrefix(path, excludedPath+"/") {
			return true
		}
	}
	return false
}
