package indexpath

import (
	"os"
	"path/filepath"
	"testing"
)

func TestGetIndexPath(t *testing.T) {
	// Create a temporary directory for testing
	tmpDir := t.TempDir()

	tests := []struct {
		name           string
		setup          func(string) error
		expectedSubdir bool // true if we expect zoekt_data to be used
		expectError    bool
		description    string
	}{
		{
			name: "empty directory creates zoekt_data",
			setup: func(dir string) error {
				return nil // directory is already empty
			},
			expectedSubdir: true,
			expectError:    false,
			description:    "should create zoekt_data subdirectory for empty base directory",
		},
		{
			name: "directory with only node.uuid creates zoekt_data",
			setup: func(dir string) error {
				return os.WriteFile(filepath.Join(dir, "node.uuid"), []byte("test-uuid"), 0o644)
			},
			expectedSubdir: true,
			expectError:    false,
			description:    "should create zoekt_data subdirectory when only special files exist",
		},
		{
			name: "directory with only repos_to_delete creates zoekt_data",
			setup: func(dir string) error {
				return os.Mkdir(filepath.Join(dir, "repos_to_delete"), 0o755)
			},
			expectedSubdir: true,
			expectError:    false,
			description:    "should create zoekt_data subdirectory when only special dirs exist",
		},
		{
			name: "directory with existing index files uses base directory",
			setup: func(dir string) error {
				return os.WriteFile(filepath.Join(dir, "test.zoekt"), []byte("test"), 0o644)
			},
			expectedSubdir: false,
			expectError:    false,
			description:    "should use base directory when it contains index files",
		},
		{
			name: "existing zoekt_data with index files is preferred",
			setup: func(dir string) error {
				// Create index file in base directory
				if err := os.WriteFile(filepath.Join(dir, "old.zoekt"), []byte("old"), 0o644); err != nil {
					return err
				}
				// Create zoekt_data subdirectory with index file
				zoektDir := filepath.Join(dir, ZoektDataDir)
				if err := os.Mkdir(zoektDir, 0o755); err != nil {
					return err
				}
				return os.WriteFile(filepath.Join(zoektDir, "new.zoekt"), []byte("new"), 0o644)
			},
			expectedSubdir: true,
			expectError:    false,
			description:    "should prefer zoekt_data subdirectory when it exists and has index files",
		},
		{
			name: "empty zoekt_data subdirectory falls back to base directory",
			setup: func(dir string) error {
				// Create index file in base directory
				if err := os.WriteFile(filepath.Join(dir, "base.zoekt"), []byte("base"), 0o644); err != nil {
					return err
				}
				// Create empty zoekt_data subdirectory
				return os.Mkdir(filepath.Join(dir, ZoektDataDir), 0o755)
			},
			expectedSubdir: false,
			expectError:    false,
			description:    "should use base directory when zoekt_data exists but is empty",
		},
		{
			name: "directory with non-index files creates zoekt_data",
			setup: func(dir string) error {
				return os.WriteFile(filepath.Join(dir, "other.txt"), []byte("content"), 0o644)
			},
			expectedSubdir: true,
			expectError:    false,
			description:    "should create zoekt_data when base directory has no index files",
		},
	}

	for _, tt := range tests {
		t.Run(tt.name, func(t *testing.T) {
			// Create a subdirectory for this test
			testDir := filepath.Join(tmpDir, tt.name)
			err := os.Mkdir(testDir, 0o755)
			if err != nil {
				t.Fatalf("Failed to create test directory: %v", err)
			}

			// Setup the test scenario
			if setupErr := tt.setup(testDir); setupErr != nil {
				t.Fatalf("Failed to setup test: %v", setupErr)
			}

			// Call GetIndexPath
			result, err := GetIndexPath(testDir)

			// Check for expected error
			if tt.expectError {
				if err == nil {
					t.Errorf("Expected error but got none")
				}
				return
			}

			if err != nil {
				t.Fatalf("Unexpected error: %v", err)
			}

			// Check the result
			expectedPath := testDir
			if tt.expectedSubdir {
				expectedPath = filepath.Join(testDir, ZoektDataDir)
			}

			if result != expectedPath {
				t.Errorf("Expected path %s, got %s. %s", expectedPath, result, tt.description)
			}

			// If we expected zoekt_data to be created, verify it exists
			if tt.expectedSubdir {
				zoektDataPath := filepath.Join(testDir, ZoektDataDir)
				if info, err := os.Stat(zoektDataPath); err != nil || !info.IsDir() {
					t.Errorf("Expected zoekt_data directory to be created at %s", zoektDataPath)
				}
			}
		})
	}
}

func TestHasIndexFiles(t *testing.T) {
	tmpDir := t.TempDir()

	tests := []struct {
		name     string
		setup    func(string) error
		expected bool
	}{
		{
			name: "directory with .zoekt file",
			setup: func(dir string) error {
				return os.WriteFile(filepath.Join(dir, "test.zoekt"), []byte("test"), 0o644)
			},
			expected: true,
		},
		{
			name: "directory without .zoekt files",
			setup: func(dir string) error {
				return os.WriteFile(filepath.Join(dir, "test.txt"), []byte("test"), 0o644)
			},
			expected: false,
		},
		{
			name: "empty directory",
			setup: func(dir string) error {
				return nil
			},
			expected: false,
		},
		{
			name: "directory with subdirectory containing .zoekt file",
			setup: func(dir string) error {
				subdir := filepath.Join(dir, "subdir")
				if err := os.Mkdir(subdir, 0o755); err != nil {
					return err
				}
				return os.WriteFile(filepath.Join(subdir, "test.zoekt"), []byte("test"), 0o644)
			},
			expected: false, // should not find files in subdirectories
		},
	}

	for _, tt := range tests {
		t.Run(tt.name, func(t *testing.T) {
			testDir := filepath.Join(tmpDir, tt.name)
			err := os.Mkdir(testDir, 0o755)
			if err != nil {
				t.Fatalf("Failed to create test directory: %v", err)
			}

			if setupErr := tt.setup(testDir); setupErr != nil {
				t.Fatalf("Failed to setup test: %v", setupErr)
			}

			result := hasIndexFiles(testDir)
			if result != tt.expected {
				t.Errorf("Expected %v, got %v", tt.expected, result)
			}
		})
	}
}
