// Package indexpath provides utilities for managing index directory paths
// with support for the new zoekt_data subdirectory structure.
package indexpath

import (
	"os"
	"path/filepath"
)

const (
	// ZoektDataDir is the name of the subdirectory for index files
	ZoektDataDir = "zoekt_data"
)

// GetIndexPath returns the appropriate index path based on the base directory.
// If a zoekt_data subdirectory exists and contains index files, use it.
// If the base directory has index files, use the base directory.
// Otherwise, create and use the zoekt_data subdirectory.
func GetIndexPath(baseDir string) (string, error) {
	zoektDataPath := filepath.Join(baseDir, ZoektDataDir)

	// Check if zoekt_data subdirectory exists
	if info, err := os.Stat(zoektDataPath); err == nil && info.IsDir() {
		// If zoekt_data exists and has content, use it
		if hasIndexFiles(zoektDataPath) {
			return zoektDataPath, nil
		}
	}

	// Check if base directory has index files
	if hasIndexFiles(baseDir) {
		return baseDir, nil
	}

	if err := os.MkdirAll(zoektDataPath, 0o750); err != nil {
		return baseDir, err
	}

	return zoektDataPath, nil
}

// hasIndexFiles checks if a directory contains any zoekt index files
func hasIndexFiles(dir string) bool {
	entries, err := os.ReadDir(dir)
	if err != nil {
		return false
	}

	for _, entry := range entries {
		if !entry.IsDir() && filepath.Ext(entry.Name()) == ".zoekt" {
			return true
		}
	}
	return false
}
