package auth

import (
	"bytes"
	"context"
	"log/slog"
	"testing"
	"time"

	"github.com/stretchr/testify/require"
	"gitlab.com/gitlab-org/gitlab-zoekt-indexer/internal/authentication"
	"google.golang.org/grpc"
	"google.golang.org/grpc/codes"
	"google.golang.org/grpc/metadata"
	"google.golang.org/grpc/status"
)

func createTestAuth() authentication.Auth {
	return authentication.NewAuth("test-issuer", time.Hour, []byte("test-secret"))
}

func createTestConfig(enabled bool) *Config {
	var logBuffer bytes.Buffer
	logger := slog.New(slog.NewJSONHandler(&logBuffer, &slog.HandlerOptions{Level: slog.LevelDebug}))

	auth := createTestAuth()
	return &Config{
		Auth:    &auth,
		Logger:  logger,
		Enabled: enabled,
	}
}

// Mock handler functions for testing
func mockUnaryHandler(ctx context.Context, req interface{}) (interface{}, error) {
	return "success", nil
}

func mockStreamHandler(srv interface{}, stream grpc.ServerStream) error {
	return nil
}

type mockServerStream struct {
	grpc.ServerStream
	ctx context.Context
}

func (m *mockServerStream) Context() context.Context {
	return m.ctx
}

func TestUnaryServerInterceptor_Disabled(t *testing.T) {
	config := createTestConfig(false)

	interceptor := config.UnaryServerInterceptor()

	// Should not do any verification when disabled
	result, err := interceptor(context.Background(), "test", &grpc.UnaryServerInfo{FullMethod: "/test"}, mockUnaryHandler)

	require.NoError(t, err)
	require.Equal(t, "success", result)
}

func TestUnaryServerInterceptor_MissingMetadata(t *testing.T) {
	config := createTestConfig(true)

	var logBuffer bytes.Buffer
	config.Logger = slog.New(slog.NewJSONHandler(&logBuffer, &slog.HandlerOptions{Level: slog.LevelDebug}))

	interceptor := config.UnaryServerInterceptor()

	// Context without metadata
	result, err := interceptor(context.Background(), "test", &grpc.UnaryServerInfo{FullMethod: "/test"}, mockUnaryHandler)

	require.Error(t, err)
	require.Nil(t, result)
	require.Equal(t, codes.Unauthenticated, status.Code(err))
	require.Contains(t, err.Error(), "missing gRPC metadata")

	// Check that warning was logged
	logOutput := logBuffer.String()
	require.Contains(t, logOutput, "Missing gRPC metadata")
}

func TestUnaryServerInterceptor_MissingAuthHeader(t *testing.T) {
	config := createTestConfig(true)

	var logBuffer bytes.Buffer
	config.Logger = slog.New(slog.NewJSONHandler(&logBuffer, &slog.HandlerOptions{Level: slog.LevelDebug}))

	interceptor := config.UnaryServerInterceptor()

	// Context with metadata but no GitlabZoektAPIRequestHeader
	md := metadata.New(map[string]string{"other": "value"})
	ctx := metadata.NewIncomingContext(context.Background(), md)

	result, err := interceptor(ctx, "test", &grpc.UnaryServerInfo{FullMethod: "/test"}, mockUnaryHandler)

	require.Error(t, err)
	require.Nil(t, result)
	require.Equal(t, codes.Unauthenticated, status.Code(err))
	require.Contains(t, err.Error(), authentication.GitlabZoektAPIRequestHeader)

	// Check that warning was logged
	logOutput := logBuffer.String()
	require.Contains(t, logOutput, "Missing authentication metadata in gRPC request")
}

func TestUnaryServerInterceptor_ValidToken(t *testing.T) {
	config := createTestConfig(true)

	var logBuffer bytes.Buffer
	config.Logger = slog.New(slog.NewJSONHandler(&logBuffer, &slog.HandlerOptions{Level: slog.LevelDebug}))

	interceptor := config.UnaryServerInterceptor()

	// Generate a valid token
	token, err := config.Auth.GenerateJWT()
	require.NoError(t, err)

	// Context with valid GitlabZoektAPIRequestHeader
	md := metadata.New(map[string]string{authentication.GitlabZoektAPIRequestHeaderLower: "Bearer " + token})
	ctx := metadata.NewIncomingContext(context.Background(), md)

	result, err := interceptor(ctx, "test", &grpc.UnaryServerInfo{FullMethod: "/test"}, mockUnaryHandler)

	require.NoError(t, err)
	require.Equal(t, "success", result)

	// Should not contain any warning logs
	logOutput := logBuffer.String()
	require.NotContains(t, logOutput, "Missing")
	require.NotContains(t, logOutput, "Invalid")
}

func TestUnaryServerInterceptor_InvalidToken(t *testing.T) {
	config := createTestConfig(true)

	var logBuffer bytes.Buffer
	config.Logger = slog.New(slog.NewJSONHandler(&logBuffer, &slog.HandlerOptions{Level: slog.LevelDebug}))

	interceptor := config.UnaryServerInterceptor()

	// Context with invalid GitlabZoektAPIRequestHeader
	md := metadata.New(map[string]string{authentication.GitlabZoektAPIRequestHeaderLower: "Bearer invalid.token.here"})
	ctx := metadata.NewIncomingContext(context.Background(), md)

	result, err := interceptor(ctx, "test", &grpc.UnaryServerInfo{FullMethod: "/test"}, mockUnaryHandler)

	require.Error(t, err)
	require.Nil(t, result)
	require.Equal(t, codes.Unauthenticated, status.Code(err))
	require.Contains(t, err.Error(), "invalid JWT token")

	// Check that warning was logged
	logOutput := logBuffer.String()
	require.Contains(t, logOutput, "Invalid JWT token in gRPC metadata")
}

func TestStreamServerInterceptor_MissingMetadata(t *testing.T) {
	config := createTestConfig(true)

	var logBuffer bytes.Buffer
	config.Logger = slog.New(slog.NewJSONHandler(&logBuffer, &slog.HandlerOptions{Level: slog.LevelDebug}))

	interceptor := config.StreamServerInterceptor()

	mockStream := &mockServerStream{ctx: context.Background()}

	err := interceptor(nil, mockStream, &grpc.StreamServerInfo{FullMethod: "/test"}, mockStreamHandler)

	require.Error(t, err)
	require.Equal(t, codes.Unauthenticated, status.Code(err))
	require.Contains(t, err.Error(), "missing gRPC metadata")

	// Check that warning was logged
	logOutput := logBuffer.String()
	require.Contains(t, logOutput, "Missing gRPC metadata")
}

func TestStreamServerInterceptor_ValidToken(t *testing.T) {
	config := createTestConfig(true)

	var logBuffer bytes.Buffer
	config.Logger = slog.New(slog.NewJSONHandler(&logBuffer, &slog.HandlerOptions{Level: slog.LevelDebug}))

	interceptor := config.StreamServerInterceptor()

	// Generate a valid token
	token, err := config.Auth.GenerateJWT()
	require.NoError(t, err)

	// Context with valid GitlabZoektAPIRequestHeader
	md := metadata.New(map[string]string{authentication.GitlabZoektAPIRequestHeaderLower: "Bearer " + token})
	ctx := metadata.NewIncomingContext(context.Background(), md)

	mockStream := &mockServerStream{ctx: ctx}

	err = interceptor(nil, mockStream, &grpc.StreamServerInfo{FullMethod: "/test"}, mockStreamHandler)

	require.NoError(t, err)

	// Should not contain any warning logs
	logOutput := logBuffer.String()
	require.NotContains(t, logOutput, "Missing")
	require.NotContains(t, logOutput, "Invalid")
}
