// Package auth provides gRPC interceptors for JWT authentication verification.
// It validates GitlabZoektAPIRequestHeader in gRPC metadata and logs warnings
// when tokens are missing or invalid.
package auth

import (
	"context"
	"log/slog"

	"gitlab.com/gitlab-org/gitlab-zoekt-indexer/internal/authentication"
	"google.golang.org/grpc"
	"google.golang.org/grpc/codes"
	"google.golang.org/grpc/metadata"
	"google.golang.org/grpc/status"
)

// Config holds the configuration for the JWT authentication interceptor
type Config struct {
	Auth    *authentication.Auth
	Logger  *slog.Logger
	Enabled bool // Whether JWT verification is enabled
}

// UnaryServerInterceptor returns a gRPC unary server interceptor that verifies JWT tokens
func (c *Config) UnaryServerInterceptor() grpc.UnaryServerInterceptor {
	return func(ctx context.Context, req interface{}, info *grpc.UnaryServerInfo, handler grpc.UnaryHandler) (interface{}, error) {
		if c.Enabled && c.Auth != nil {
			if err := c.verifyJWTFromMetadata(ctx, info.FullMethod); err != nil {
				return nil, err
			}
		}
		return handler(ctx, req)
	}
}

// StreamServerInterceptor returns a gRPC streaming server interceptor that verifies JWT tokens
func (c *Config) StreamServerInterceptor() grpc.StreamServerInterceptor {
	return func(srv interface{}, ss grpc.ServerStream, info *grpc.StreamServerInfo, handler grpc.StreamHandler) error {
		if c.Enabled && c.Auth != nil {
			if err := c.verifyJWTFromMetadata(ss.Context(), info.FullMethod); err != nil {
				return err
			}
		}
		return handler(srv, ss)
	}
}

// verifyJWTFromMetadata extracts and verifies JWT token from gRPC metadata
func (c *Config) verifyJWTFromMetadata(ctx context.Context, method string) error {
	md, ok := metadata.FromIncomingContext(ctx)
	if !ok {
		c.Logger.Warn("Missing gRPC metadata", "method", method)
		return status.Errorf(codes.Unauthenticated, "missing gRPC metadata")
	}

	// Get GitlabZoektAPIRequestHeader from metadata
	authHeaders := md.Get(authentication.GitlabZoektAPIRequestHeaderLower)
	if len(authHeaders) == 0 {
		c.Logger.Warn("Missing authentication metadata in gRPC request", "header", authentication.GitlabZoektAPIRequestHeader, "method", method)
		return status.Errorf(codes.Unauthenticated, "missing %s header", authentication.GitlabZoektAPIRequestHeader)
	}

	// Use the first authorization header
	authHeader := authHeaders[0]

	// Verify the Bearer token
	_, err := c.Auth.VerifyBearerToken(authHeader)
	if err != nil {
		c.Logger.Warn("Invalid JWT token in gRPC metadata", "method", method, "error", err.Error())
		return status.Errorf(codes.Unauthenticated, "invalid JWT token: %v", err)
	}

	// Token is valid
	return nil
}
