package convert_test

import (
	"math"
	"testing"

	"github.com/stretchr/testify/require"
	"gitlab.com/gitlab-org/gitlab-zoekt-indexer/internal/convert"
)

func TestSafeConvertUint64ToUint32(t *testing.T) {
	tests := []struct {
		name string
		id   uint64
		want uint32
	}{
		{"within uint32 range", 12345, 12345},
		{"zero", 0, 0},
		{"max uint32", math.MaxUint32, math.MaxUint32},
		{"outside of uint32 range", math.MaxUint32 + 1, 0},
	}

	for _, tt := range tests {
		t.Run(tt.name, func(t *testing.T) {
			got := convert.SafeConvertUint64ToUint32(tt.id)
			if got != tt.want {
				t.Errorf("SafeConvertUint64ToUint32(%d) = %d, want %d", tt.id, got, tt.want)
			}
		})
	}
}

func TestToSHA256(t *testing.T) {
	cases := []struct {
		name     string
		input    any
		expected string
	}{
		{
			name:     "simple string",
			input:    "hello world",
			expected: "9ddefe4435b21d901439e546d54a14a175a3493b9fd8fbf38d9ea6d3cbf70826", // sha256 of '"hello world"' (with quotes, as JSON)
		},
		{
			name:     "simple map",
			input:    map[string]string{"foo": "bar"},
			expected: "7a38bf81f383f69433ad6e900d35b3e2385593f76a7b7ab5d4355b8ba41ee24b", // sha256 of '{"foo":"bar"}'
		},
		{
			name:     "int array",
			input:    []int{1, 2, 3},
			expected: "a615eeaee21de5179de080de8c3052c8da901138406ba71c38c032845f7d54f4", // sha256 of '[1,2,3]'
		},
		{
			name: "json object",
			input: struct {
				A int
				B string
			}{A: 1, B: "x"},
			expected: "0f3656c3fd7ee6de73723072536d83aaeb349d3a6ac53ea704fcb07c7329f07e", // sha256 of '{"A":1,"B":"x"}'
		},
	}

	for _, c := range cases {
		t.Run(c.name, func(t *testing.T) {
			got, err := convert.ToSHA256(c.input)
			require.NoError(t, err)

			if got != c.expected {
				t.Errorf("ToSHA256(%v) = %s, want %s", c.input, got, c.expected)
			}
		})
	}
}
