// Package convert provides utility functions for safe type conversions.
package convert

import (
	"crypto/sha256"
	"encoding/hex"
	"encoding/json"
	"log/slog"
	"math"
)

// SafeConvertUint64ToUint32 safely converts a uint64 to uint32.
// If the value exceeds the maximum uint32 value, it returns 0.
func SafeConvertUint64ToUint32(value uint64) uint32 {
	if value > math.MaxUint32 {
		slog.Debug("value exceeds maximum uint32 value, returning 0", "value", value)

		return 0
	}
	return uint32(value)
}

// ToSHA256 takes any value, marshals it to JSON, and returns the SHA256 hash of the JSON bytes as a hex string.
func ToSHA256(v any) (string, error) {
	data, err := json.Marshal(v)
	if err != nil {
		return "", err
	}
	h := sha256.New()
	h.Write(data)
	return hex.EncodeToString(h.Sum(nil)), nil
}
