package callback_test

import (
	"context"
	"encoding/json"
	"errors"
	"fmt"
	"io"
	"net/http"
	"net/http/httptest"
	"net/url"
	"os"
	"path/filepath"
	"testing"

	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/require"
	"gitlab.com/gitlab-org/gitlab-zoekt-indexer/internal/callback"
	"gitlab.com/gitlab-org/gitlab-zoekt-indexer/internal/secretreader"
	"go.uber.org/goleak"
)

const (
	nodeUUID = "3869fe21-36d1"
)

func TestSuccessCallbackRequestForIndex(t *testing.T) {
	indexDir := t.TempDir()
	data := []byte("secret")
	err := os.WriteFile(filepath.Join(indexDir, "secret.txt"), data, 0644)
	require.NoError(t, err)
	var requestedURL url.URL
	var requestedBody interface{}
	svr := stubServer(&requestedURL, &requestedBody)
	defer svr.Close()
	callbackAPIInstance, err := callbackAPIInstance(svr.URL, indexDir)
	require.NoError(t, err)
	callbackAPIInstance.SendSuccess(context.Background(), defaultCallbackParams(), indexDir, 1)
	expectedPath := fmt.Sprintf("/api/v4/internal/search/zoekt/%s/callback", url.PathEscape(nodeUUID))
	require.Equal(t, expectedPath, requestedURL.Path)
	requestedBodyMap := requestedBody.(map[string]interface{})
	require.Equal(t, requestedBodyMap["name"], defaultCallbackParams().Name)
	require.True(t, requestedBodyMap["success"].(bool))
	assert.Contains(t, requestedBodyMap, "additional_payload")
	assert.Contains(t, requestedBodyMap["additional_payload"], "repo_stats")
	repoStatsMap := requestedBodyMap["additional_payload"].(map[string]interface{})["repo_stats"].(map[string]interface{})
	assert.Contains(t, repoStatsMap, "size_in_bytes")
	assert.Contains(t, repoStatsMap, "index_file_count")
	assert.Contains(t, requestedBodyMap["payload"], "key")
}

func TestFailureCallbackRequestForIndex(t *testing.T) {
	indexDir := t.TempDir()
	data := []byte("secret")
	err := os.WriteFile(filepath.Join(indexDir, "secret.txt"), data, 0644)
	require.NoError(t, err)

	var requestedURL url.URL
	var requestedBody interface{}
	svr := stubServer(&requestedURL, &requestedBody)
	defer svr.Close()
	callbackAPIInstance, err := callbackAPIInstance(svr.URL, indexDir)
	require.NoError(t, err)
	failureReason := errors.New("Sample Error")
	callbackAPIInstance.SendFailure(context.Background(), defaultCallbackParams(), indexDir, 1, failureReason)
	expectedPath := fmt.Sprintf("/api/v4/internal/search/zoekt/%s/callback", url.PathEscape(nodeUUID))
	require.Equal(t, expectedPath, requestedURL.Path)
	requestedBodyMap := requestedBody.(map[string]interface{})
	require.Equal(t, requestedBodyMap["name"], defaultCallbackParams().Name)
	require.False(t, requestedBodyMap["success"].(bool))
	require.Equal(t, requestedBodyMap["error"], failureReason.Error())
	assert.Contains(t, requestedBodyMap, "additional_payload")
	assert.Contains(t, requestedBodyMap["additional_payload"], "repo_stats")
	repoStatsMap := requestedBodyMap["additional_payload"].(map[string]interface{})["repo_stats"].(map[string]interface{})
	assert.Contains(t, repoStatsMap, "size_in_bytes")
	assert.Contains(t, repoStatsMap, "index_file_count")
	assert.Contains(t, requestedBodyMap["payload"], "key")
}

func TestNewCallbackAPIWithSecret(t *testing.T) {
	var requestedURL url.URL
	var requestedBody interface{}
	svr := stubServer(&requestedURL, &requestedBody)
	defer svr.Close()

	secret := []byte("test-secret")
	api, err := callback.NewCallbackAPI(svr.URL, nodeUUID, secret, &http.Client{})

	require.NoError(t, err)
	require.NotNil(t, api)
	require.Equal(t, secret, api.Secret)
}

func callbackAPIInstance(internalURL string, indexDir string) (callback.CallbackAPI, error) {
	// Read secret from file and trim whitespace using the secretreader package
	secretPath := filepath.Join(indexDir, "secret.txt")
	secret, err := secretreader.ReadSecret(secretPath)
	if err != nil {
		return callback.CallbackAPI{}, err
	}

	return callback.NewCallbackAPI(internalURL, nodeUUID, secret, &http.Client{})
}

func defaultCallbackParams() callback.CallbackParams {
	return callback.CallbackParams{
		Name: "index",
		RailsPayload: map[string]interface{}{
			"key": "value",
		},
	}
}

func stubServer(svrRequestedPath *url.URL, requestedBody interface{}) *httptest.Server {
	svr := httptest.NewServer(http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
		*svrRequestedPath = *r.URL
		buf, _ := io.ReadAll(r.Body)
		err := json.Unmarshal(buf, &requestedBody)
		if err != nil {
			panic(err)
		}

	}))
	return svr
}

func TestMain(m *testing.M) {
	goleak.VerifyTestMain(m)
}
