package authentication

import (
	"testing"
	"time"

	"github.com/stretchr/testify/require"
	"go.uber.org/goleak"
)

const (
	staticSecret = "foo"
	jwtIssuer    = "gitlab-shell"
	jwtTTL       = time.Minute
)

func defaultStubbedAuth() Auth {
	timeNowFunc := func() time.Time {
		return time.Date(2000, time.January, 19, 1, 2, 3, 4, time.UTC)
	}

	return Auth{
		jwtIssuer:   jwtIssuer,
		jwtTTL:      jwtTTL,
		secretBytes: []byte(staticSecret),
		timeNowFunc: timeNowFunc,
	}
}

func TestGenerateToken(t *testing.T) {
	tokenString, err := defaultStubbedAuth().GenerateJWT()

	require.NoError(t, err)
	expectedToken := "eyJhbGciOiJIUzI1NiIsInR5cCI6IkpXVCJ9.eyJpc3MiOiJnaXRsYWItc2hlbGwiLCJleHAiOjk0ODI0Mzc4MywiaWF0Ijo5NDgyNDM3MjN9._A8tB69mW7tXCdzwdVuqI6lPSBtKkAhH0nmbXi3K6tk" // gitleaks:allow
	require.Equal(t, expectedToken, tokenString)
}

func TestVerifyJWT(t *testing.T) {
	auth := defaultStubbedAuth()

	// Generate a token and then verify it
	tokenString, err := auth.GenerateJWT()
	require.NoError(t, err)

	claims, err := auth.verifyJWT(tokenString)
	require.NoError(t, err)
	require.NotNil(t, claims)
	require.Equal(t, jwtIssuer, claims.Issuer)
	require.NotNil(t, claims.IssuedAt)
	require.NotNil(t, claims.ExpiresAt)
}

func TestVerifyJWT_EmptyToken(t *testing.T) {
	auth := defaultStubbedAuth()

	claims, err := auth.verifyJWT("")
	require.Error(t, err)
	require.Nil(t, claims)
	require.Contains(t, err.Error(), "empty token")
}

func TestVerifyJWT_InvalidToken(t *testing.T) {
	auth := defaultStubbedAuth()

	claims, err := auth.verifyJWT("invalid.token.here")
	require.Error(t, err)
	require.Nil(t, claims)
	require.Contains(t, err.Error(), "failed to parse token")
}

func TestVerifyJWT_WrongSecret(t *testing.T) {
	auth := defaultStubbedAuth()

	// Generate token with correct secret
	tokenString, err := auth.GenerateJWT()
	require.NoError(t, err)

	// Try to verify with wrong secret
	wrongAuth := Auth{
		jwtIssuer:   jwtIssuer,
		jwtTTL:      jwtTTL,
		secretBytes: []byte("wrong-secret"),
		timeNowFunc: auth.timeNowFunc,
	}

	claims, err := wrongAuth.verifyJWT(tokenString)
	require.Error(t, err)
	require.Nil(t, claims)
	require.Contains(t, err.Error(), "failed to parse token")
}

func TestVerifyBearerToken(t *testing.T) {
	auth := defaultStubbedAuth()

	// Generate a token
	tokenString, err := auth.GenerateJWT()
	require.NoError(t, err)

	// Test valid Bearer token
	authHeader := "Bearer " + tokenString
	claims, err := auth.VerifyBearerToken(authHeader)
	require.NoError(t, err)
	require.NotNil(t, claims)
	require.Equal(t, jwtIssuer, claims.Issuer)
}

func TestVerifyBearerToken_EmptyHeader(t *testing.T) {
	auth := defaultStubbedAuth()

	claims, err := auth.VerifyBearerToken("")
	require.Error(t, err)
	require.Nil(t, claims)
	require.Contains(t, err.Error(), "missing authentication header: "+GitlabZoektAPIRequestHeader)
}

func TestVerifyBearerToken_NoBearerPrefix(t *testing.T) {
	auth := defaultStubbedAuth()

	tokenString, err := auth.GenerateJWT()
	require.NoError(t, err)

	// Test without Bearer prefix
	claims, err := auth.VerifyBearerToken(tokenString)
	require.Error(t, err)
	require.Nil(t, claims)
	require.Contains(t, err.Error(), "authorization header must start with 'Bearer '")
}

func TestMain(m *testing.M) {
	goleak.VerifyTestMain(m)
}
