// Package authentication provides JWT token generation functionality.
// It implements a simple authentication mechanism using HS256-signed JWTs
// with configurable issuer, expiration time, and signing secret.
// The package uses github.com/golang-jwt/jwt/v5 for token creation and signing.
package authentication

import (
	"errors"
	"fmt"
	"strings"
	"time"

	"github.com/golang-jwt/jwt/v5"
)

const (
	GitlabIssuer = "gitlab"
	// GitlabZoektAPIRequestHeader is the header name used for JWT authentication
	GitlabZoektAPIRequestHeader = "Gitlab-Zoekt-Api-Request"
	// GitlabZoektAPIRequestHeaderLower is the lowercase version used in gRPC metadata
	GitlabZoektAPIRequestHeaderLower = "gitlab-zoekt-api-request"
)

type Auth struct {
	jwtIssuer   string
	jwtTTL      time.Duration
	secretBytes []byte
	timeNowFunc func() time.Time
}

func NewAuth(jwtIssuer string, jwtTTL time.Duration, secret []byte) Auth {
	return Auth{
		jwtIssuer:   jwtIssuer,
		jwtTTL:      jwtTTL,
		secretBytes: secret,
		timeNowFunc: time.Now,
	}
}

func (a Auth) GenerateJWT() (string, error) {
	claims := jwt.RegisteredClaims{
		Issuer:    a.jwtIssuer,
		IssuedAt:  jwt.NewNumericDate(a.timeNowFunc()),
		ExpiresAt: jwt.NewNumericDate(a.timeNowFunc().Add(a.jwtTTL)),
	}
	return jwt.NewWithClaims(jwt.SigningMethodHS256, claims).SignedString(a.secretBytes)
}

// verifyJWT verifies a JWT token string and returns the claims if valid
func (a Auth) verifyJWT(tokenString string) (*jwt.RegisteredClaims, error) {
	if tokenString == "" {
		return nil, errors.New("empty token")
	}

	// Create parser with custom time function
	parser := jwt.NewParser(jwt.WithTimeFunc(a.timeNowFunc))

	token, err := parser.ParseWithClaims(tokenString, &jwt.RegisteredClaims{}, func(token *jwt.Token) (interface{}, error) {
		// Validate the signing method
		if _, ok := token.Method.(*jwt.SigningMethodHMAC); !ok {
			return nil, fmt.Errorf("unexpected signing method: %v", token.Header["alg"])
		}
		return a.secretBytes, nil
	})

	if err != nil {
		return nil, fmt.Errorf("failed to parse token: %w", err)
	}

	claims, ok := token.Claims.(*jwt.RegisteredClaims)
	if !ok || !token.Valid {
		return nil, errors.New("invalid token claims")
	}

	// Verify issuer if configured
	if a.jwtIssuer != "" && claims.Issuer != a.jwtIssuer {
		return nil, fmt.Errorf("invalid issuer: expected %s, got %s", a.jwtIssuer, claims.Issuer)
	}

	// Verify expiration time
	if claims.ExpiresAt != nil && claims.ExpiresAt.Before(a.timeNowFunc()) {
		return nil, fmt.Errorf("token expired at %v", claims.ExpiresAt)
	}

	return claims, nil
}

// VerifyBearerToken extracts and verifies a Bearer token from GitlabZoektAPIRequestHeader
func (a Auth) VerifyBearerToken(authHeader string) (*jwt.RegisteredClaims, error) {
	if authHeader == "" {
		return nil, fmt.Errorf("missing authentication header: %s", GitlabZoektAPIRequestHeader)
	}

	const bearerPrefix = "Bearer "
	if !strings.HasPrefix(authHeader, bearerPrefix) {
		return nil, errors.New("authorization header must start with 'Bearer '")
	}

	token := strings.TrimPrefix(authHeader, bearerPrefix)
	return a.verifyJWT(token)
}
