package main

import (
	"flag"
	"fmt"
	"log/slog"
	"os"
	"strings"

	"gitlab.com/gitlab-org/gitlab-zoekt-indexer/internal/mode/indexer"
	"gitlab.com/gitlab-org/gitlab-zoekt-indexer/internal/mode/webserver"
)

var (
	// Overriden in the makefile
	Version   = "dev"
	BuildTime = ""
)

func printHelp() {
	fmt.Fprintf(os.Stderr, `Usage: %s <command> [options]

Commands:
  indexer     Run in indexer mode
  webserver   Run in webserver mode
  version     Print version information

For command specific help:
  %s <command> -help

`, os.Args[0], os.Args[0])
}

func main() {
	if len(os.Args) < 2 {
		printHelp()
		os.Exit(1)
	}

	command := os.Args[1]

	// Handle help flags at the top level
	if command == "help" || command == "-help" || command == "--help" || command == "-h" {
		printHelp()
		os.Exit(0)
	}

	// Handle version command at the top level
	if command == "version" {
		fmt.Printf("%s %s (built at: %s)\n", os.Args[0], Version, BuildTime) // nolint:forbidigo
		os.Exit(0)
	}

	// Shift the arguments to allow flags to be parsed correctly for subcommands
	// This makes it so that gitlab-zoekt indexer -index_dir=/data/index looks like
	// the original gitlab-zoekt-indexer -index_dir=/data/index
	//
	// We keep the original program name (argv[0]) but pass all other arguments
	// (including version flags) to the underlying mode implementation
	shiftedArgs := []string{os.Args[0]}
	if len(os.Args) > 2 {
		shiftedArgs = append(shiftedArgs, os.Args[2:]...)
	}
	os.Args = shiftedArgs

	// Reset flag parse state
	flag.CommandLine = flag.NewFlagSet(os.Args[0], flag.ExitOnError)

	// Run the appropriate command
	switch strings.ToLower(command) {
	case "indexer":
		err := runIndexer()
		if err != nil {
			slog.Error("program aborted", "error", err)
			os.Exit(1)
		}
	case "webserver":
		err := runWebserver()
		if err != nil {
			slog.Error("program aborted", "error", err)
			os.Exit(1)
		}
	default:
		fmt.Fprintf(os.Stderr, "Unknown command: %s\n\n", command)
		printHelp()
		os.Exit(1)
	}
}

func runIndexer() error {
	opts, err := indexer.ParseFlags()
	if err != nil {
		return err
	}
	return indexer.Run(opts, Version, BuildTime)
}

func runWebserver() error {
	opts, err := webserver.ParseFlags()
	if err != nil {
		return err
	}
	return webserver.Run(opts, Version, BuildTime)
}
