#!/bin/bash

set -euo pipefail

# Get the Go version from our go.mod file
LOCAL_GO_VERSION=$(grep -E "^go [0-9]+\.[0-9]+(\.[0-9]+)?" go.mod | awk '{print $2}')

echo "Local go.mod Go version: ${LOCAL_GO_VERSION}"

# Helper functions for version comparison
# ======================================

# Parse version to format suitable for numeric comparison
parse_version() {
  echo "$1" | awk -F. '{ printf("%d%03d%03d\n", $1, $2, $3); }'
}


# Add .0 suffix if the version doesn't include patch version
normalize_version() {
  local version="$1"
  if [[ "$version" =~ ^[0-9]+\.[0-9]+$ ]]; then
    echo "${version}.0"
  else
    echo "$version"
  fi
}

# Compare local version with remote version
# Returns 0 if versions are compatible, 1 otherwise
compare_versions() {
  local source_name="$1"
  local remote_version="$2"
  local local_version="${LOCAL_GO_VERSION}"
  
  echo "${source_name} GO_VERSION: ${remote_version}"
  
  NORMALIZED_LOCAL_VERSION=$(normalize_version "${local_version}")
  NORMALIZED_REMOTE_VERSION=$(normalize_version "${remote_version}")
  
  LOCAL_VERSION_INT=$(parse_version "${NORMALIZED_LOCAL_VERSION}")
  REMOTE_VERSION_INT=$(parse_version "${NORMALIZED_REMOTE_VERSION}")
  
  # Only fail if local version is newer than remote version
  if [ "${LOCAL_VERSION_INT}" -gt "${REMOTE_VERSION_INT}" ]; then
    echo "Error: Local go.mod version (${local_version}) is newer than ${source_name} GO_VERSION (${remote_version})"
    echo "Please update the ${source_name} Go version or downgrade the local go.mod version"
    return 1
  else
    echo "✅ ${source_name} GO_VERSION (${remote_version}) is compatible with local go.mod version (${local_version})"
    return 0
  fi
}

# Check CNG repository
# ======================================
echo "Fetching GO_VERSION from CNG repository..."
VARIABLES_YML_URL="https://gitlab.com/gitlab-org/build/CNG/-/raw/master/ci_files/variables.yml"
VARIABLES_YML=$(curl -s --fail "${VARIABLES_YML_URL}" || { echo "Failed to download ${VARIABLES_YML_URL}"; exit 1; })

# Extract GO_VERSION from the variables.yml file
CNG_GO_VERSION=$(echo "${VARIABLES_YML}" | grep -E "^\s*GO_VERSION:" | awk '{print $2}' | tr -d '"')

if [ -z "${CNG_GO_VERSION}" ]; then
  echo "Error: Failed to fetch GO_VERSION from CNG repository"
  exit 1
fi

# Check omnibus-builder repository
# ======================================
echo "Fetching GO_VERSION from gitlab-omnibus-builder repository..."
VERSIONS_URL="https://gitlab.com/gitlab-org/gitlab-omnibus-builder/-/raw/master/docker/VERSIONS"
VERSIONS_CONTENT=$(curl -s --fail "${VERSIONS_URL}" || { echo "Failed to download ${VERSIONS_URL}"; exit 1; })

# Extract GO_VERSION from the VERSIONS file
# Assuming the format is similar to GO_VERSION=x.y.z or GO_VERSION="x.y.z"
OMNIBUS_GO_VERSION=$(echo "${VERSIONS_CONTENT}" | grep -E "^\s*GO_VERSION=" | cut -d'=' -f2 | tr -d '"' | tr -d ' ')

if [ -z "${OMNIBUS_GO_VERSION}" ]; then
  echo "Error: Failed to fetch GO_VERSION from gitlab-omnibus-builder repository"
  exit 1
fi

# Compare versions
# ======================================
CNG_RESULT=0
OMNIBUS_RESULT=0

compare_versions "CNG" "${CNG_GO_VERSION}" || CNG_RESULT=$?
compare_versions "Omnibus" "${OMNIBUS_GO_VERSION}" || OMNIBUS_RESULT=$?

# Exit with error if any comparison failed
if [ $CNG_RESULT -ne 0 ] || [ $OMNIBUS_RESULT -ne 0 ]; then
  exit 1
fi

# If we get here, all checks passed
echo "✅ All Go version checks passed"
exit 0