package main

import (
	"errors"
	"log/slog"
	"os"

	"gitlab.com/gitlab-org/gitlab-elasticsearch-indexer/internal/dev_debug"
	"gitlab.com/gitlab-org/gitlab-elasticsearch-indexer/internal/mode/advanced"
	"gitlab.com/gitlab-org/gitlab-elasticsearch-indexer/internal/mode/chunk"
	"gitlab.com/gitlab-org/gitlab-elasticsearch-indexer/internal/shared"
)

const (
	ModeEnvVar = "GITLAB_INDEXER_MODE"

	ModeAdvanced = "advanced"
	ModeChunk    = "chunk"

	DefaultMode = ModeAdvanced
)

var (
	// Overriden in the makefile
	Version   = "dev"
	BuildTime = ""
)

func main() {
	setUpLogger()

	currentMode := getMode()
	slog.Debug("Selected mode", "mode", currentMode)

	buildOpts := shared.BuildOpts{
		Version:   Version,
		BuildTime: BuildTime,
	}

	switch currentMode {
	case ModeAdvanced:
		if err := advanced.Run(buildOpts); err != nil {
			slog.Error("program aborted", "error", err)
			os.Exit(1)
		}
	case ModeChunk:
		if err := chunk.Run(buildOpts); err != nil {
			slog.Error("program aborted", "error", err)
			os.Exit(1)
		}
	default:
		err := errors.New("unsupported mode: " + currentMode)
		slog.Error(err.Error())
		os.Exit(1)
	}
}

func getMode() string {
	mode := os.Getenv(ModeEnvVar)
	if mode == "" {
		mode = DefaultMode
	}

	return mode
}

func setUpLogger() {
	level := slog.LevelInfo

	if dev_debug.IsEnabled() {
		level = slog.LevelDebug
	}

	logger := slog.New(slog.NewJSONHandler(os.Stderr, &slog.HandlerOptions{Level: level}))
	slog.SetDefault(logger)
}
