package binary

import (
	"bytes"
	"testing"

	"github.com/stretchr/testify/require"
)

func TestDetectBinary(t *testing.T) {
	t.Parallel()

	tests := []struct {
		name     string
		filename string
		data     []byte
		expected bool
	}{
		{
			name:     "empty data",
			filename: "test.txt",
			data:     []byte{},
			expected: false,
		},
		{
			name:     "single null byte",
			filename: "test.txt",
			data:     []byte{0},
			expected: true,
		},
		{
			name:     "plain text",
			filename: "test.txt",
			data:     []byte("Hello, World!"),
			expected: false,
		},
		{
			name:     "utf8 with special characters",
			filename: "test.txt",
			data:     []byte("Hello 世界! 🌍 Ñoël"),
			expected: false,
		},
		{
			name:     "null byte at start",
			filename: "test.txt",
			data:     []byte{0, 'a', 'b', 'c'},
			expected: true,
		},
		{
			name:     "null byte in middle",
			filename: "test.txt",
			data:     []byte("Hello\x00World"),
			expected: true,
		},
		{
			name:     "null byte at end within limit",
			filename: "test.txt",
			data:     []byte("Hello World\x00"),
			expected: true,
		},
		{
			name:     "null byte at search limit boundary",
			filename: "test.txt",
			data:     append(bytes.Repeat([]byte("a"), binarySearchLimit-1), 0),
			expected: true,
		},
		{
			name:     "null byte beyond search limit",
			filename: "test.txt",
			data:     append(bytes.Repeat([]byte("a"), binarySearchLimit), 0),
			expected: false,
		},
		{
			name:     "large file without null bytes",
			filename: "test.txt",
			data:     bytes.Repeat([]byte("text"), 3*1024), // 12KB of text
			expected: false,
		},
		{
			name:     "svg file by extension lowercase",
			filename: "image.svg",
			data:     []byte(`<?xml version="1.0"?><svg xmlns="http://www.w3.org/2000/svg"><circle r="50"/></svg>`),
			expected: true,
		},
		{
			name:     "svg file by extension uppercase",
			filename: "IMAGE.SVG",
			data:     []byte(`<?xml version="1.0"?><SVG xmlns="http://www.w3.org/2000/svg"><circle r="50"/></SVG>`),
			expected: true,
		},
		{
			name:     "pdf file",
			filename: "document.pdf",
			data:     []byte("some pdf content"),
			expected: true,
		},
		{
			name:     "zip file",
			filename: "archive.zip",
			data:     []byte("some zip content"),
			expected: true,
		},
		{
			name:     "executable file",
			filename: "program.exe",
			data:     []byte("some executable content"),
			expected: true,
		},
		{
			name:     "text file mentioning svg",
			filename: "README",
			data:     []byte("This document explains how to use svg graphics in your project"),
			expected: false,
		},
		{
			name:     "go file with svg variable",
			filename: "main.go",
			data:     []byte(`package main\nvar svgContent = "..."'`),
			expected: false,
		},
	}

	for _, tt := range tests {
		t.Run(tt.name, func(t *testing.T) {
			t.Parallel()

			result := DetectBinary(tt.filename, tt.data)
			require.Equal(t, tt.expected, result)
		})
	}
}

func TestDetectBinarySearchLimit(t *testing.T) {
	t.Parallel()

	require.Equal(t, 8*1024, binarySearchLimit, "binarySearchLimit should be 8KB")
}
