package indexer

import (
	"context"
	"crypto/sha256"
	"fmt"

	"gitlab.com/gitlab-org/gitlab-elasticsearch-indexer/internal/mode/chunk/chunker"
)

type IndexingStrategy interface {
	Index(ctx context.Context, projectID uint64, chunks []chunker.Chunk) ([]string, error)
	DeletePaths(ctx context.Context, projectID uint64, paths []string) error
	ResolveReindexing(ctx context.Context, projectID uint64) error
	Flush(ctx context.Context) ([]string, error)
	Close(ctx context.Context) error
}

type ChunkDocument struct {
	ID         string              `json:"id"`
	ProjectID  uint64              `json:"project_id"`
	Path       string              `json:"path"`
	Type       chunker.ContentType `json:"type"`
	Content    string              `json:"content"`
	Name       string              `json:"name"`
	Source     string              `json:"source"`
	Language   string              `json:"language"`
	Reindexing bool                `json:"reindexing"`
}

func BuildChunkDocument(chunk chunker.Chunk, projectID uint64, reindexing bool) ChunkDocument {
	id := GenerateChunkID(projectID, chunk.Path, chunk.Content)

	return ChunkDocument{
		ID:         id,
		ProjectID:  projectID,
		Path:       chunk.Path,
		Type:       chunk.Type,
		Content:    chunk.Content,
		Name:       chunk.Name,
		Source:     chunk.GetSource(),
		Language:   chunk.Language,
		Reindexing: reindexing,
	}
}

func GenerateChunkID(projectID uint64, path string, content string) string {
	data := fmt.Sprintf("%d:%s:%s", projectID, path, content)
	return fmt.Sprintf("%x", sha256.Sum256([]byte(data)))
}
