package elasticsearch_test

import (
	"context"
	"encoding/json"
	"os"
	"testing"

	"github.com/stretchr/testify/require"
	"gitlab.com/gitlab-org/gitlab-elasticsearch-indexer/internal/mode/chunk/chunker"
	"gitlab.com/gitlab-org/gitlab-elasticsearch-indexer/internal/mode/chunk/client/elasticsearch"
	"gitlab.com/gitlab-org/gitlab-elasticsearch-indexer/internal/mode/chunk/indexer"
	elasticsearchIndexer "gitlab.com/gitlab-org/gitlab-elasticsearch-indexer/internal/mode/chunk/indexer/elasticsearch"
	"gitlab.com/gitlab-org/gitlab-elasticsearch-indexer/internal/mode/chunk/types"
)

const (
	testProjectID = uint64(12345)
	testIndexName = "test-chunks"
)

// parseElasticConnectionInfo parses the ELASTIC_CONNECTION_INFO environment variable
// and skips the test if the environment variable is not set
func parseElasticConnectionInfo(t *testing.T) *types.ElasticsearchConnection {
	connInfo := os.Getenv("ELASTIC_CONNECTION_INFO")
	if connInfo == "" {
		t.Skip("ELASTIC_CONNECTION_INFO not set")
	}

	var config types.ElasticsearchConnection
	err := json.Unmarshal([]byte(connInfo), &config)
	require.NoError(t, err, "failed to parse ELASTIC_CONNECTION_INFO")

	return &config
}

func setupTestClient(t *testing.T) *elasticsearch.ElasticsearchClient {
	config := parseElasticConnectionInfo(t)
	client := elasticsearch.New(*config)
	require.NoError(t, client.Connect(context.Background()))

	return client
}

func setupTestIndexer(t *testing.T) (*elasticsearchIndexer.Indexer, *elasticsearch.ElasticsearchClient) {
	client := setupTestClient(t)
	ctx := context.Background()

	mapping := `{
		"mappings": {
			"properties": {
				"path": {"type": "keyword"},
				"project_id": {"type": "long"},
				"reindexing": {"type": "boolean"}
			}
		}
	}`

	client.Client.DeleteIndex(testIndexName).Do(ctx)
	_, err := client.Client.CreateIndex(testIndexName).BodyString(mapping).Do(ctx)
	require.NoError(t, err)

	esIndexer, err := elasticsearchIndexer.New(client, testIndexName, 2, false)
	require.NoError(t, err)

	return esIndexer, client
}

func queryAllDocuments(t *testing.T, client *elasticsearch.ElasticsearchClient) []indexer.ChunkDocument {
	ctx := context.Background()

	client.Client.Refresh(testIndexName).Do(ctx)

	// Query all documents in the index
	searchResult, err := client.Client.Search().
		Index(testIndexName).
		Do(ctx)

	require.NoError(t, err, "failed to search all documents")

	documents := make([]indexer.ChunkDocument, len(searchResult.Hits.Hits))
	for i, hit := range searchResult.Hits.Hits {
		var doc indexer.ChunkDocument
		err := json.Unmarshal(hit.Source, &doc)
		require.NoError(t, err, "failed to unmarshal document")

		documents[i] = doc
	}

	return documents
}

func verifyDocumentsContent(t *testing.T, expectedChunks []chunker.Chunk, actualDocuments []indexer.ChunkDocument) {
	require.Len(t, expectedChunks, len(actualDocuments), "document count mismatch")

	// Extract actual chunk content from search results
	actualContents := make([]string, len(actualDocuments))
	for i, doc := range actualDocuments {
		actualContents[i] = doc.Content
	}

	// Extract expected content
	expectedContents := make([]string, len(expectedChunks))
	for i, chunk := range expectedChunks {
		expectedContents[i] = chunk.Content
	}

	// Compare the content sets
	require.ElementsMatch(t, expectedContents, actualContents, "chunk contents don't match expected set")
}

// verifyElasticFinalChunks queries all documents in the index
// and verifies we have exactly the expected chunk content
func verifyElasticFinalChunks(t *testing.T, client *elasticsearch.ElasticsearchClient, expectedChunks []chunker.Chunk) {
	documents := queryAllDocuments(t, client)
	verifyDocumentsContent(t, expectedChunks, documents)
}

func TestNew(t *testing.T) {
	client := setupTestClient(t)

	esIndexer, err := elasticsearchIndexer.New(client, testIndexName, 2, false)
	require.NoError(t, err)
	require.NotNil(t, esIndexer)
}

func TestNewWithNotConnectedClient(t *testing.T) {
	config := parseElasticConnectionInfo(t)
	client := elasticsearch.New(*config)
	// We intentionally don't call client.Connect() here to test the error case

	_, err := elasticsearchIndexer.New(client, testIndexName, 2, false)
	require.Error(t, err)
	require.Contains(t, err.Error(), "elasticsearch client not connected")
}

func TestIndex(t *testing.T) {
	esIndexer, client := setupTestIndexer(t)
	ctx := context.Background()

	chunks := []chunker.Chunk{
		indexer.CreateTestChunk("file1.txt", "First chunk", "abc123", 0),
		indexer.CreateTestChunk("file1.txt", "Second chunk", "abc123", 8),
		indexer.CreateTestChunk("file2.txt", "Another file", "def456", 16),
	}

	indexedIDs, err := esIndexer.Index(ctx, testProjectID, chunks)
	require.NoError(t, err)
	require.Len(t, indexedIDs, 3)

	// Verify each chunk has the expected ID
	for i, chunk := range chunks {
		expectedID := indexer.GenerateChunkID(testProjectID, chunk.Path, chunk.Content)
		require.Equal(t, expectedID, indexedIDs[i])
	}

	// Verify we have the chunks
	verifyElasticFinalChunks(t, client, chunks)
}

func TestDeleteOrphanedChunks(t *testing.T) {
	esIndexer, client := setupTestIndexer(t)
	ctx := context.Background()

	// Initial state: Index a file with 3 chunks
	originalChunks := []chunker.Chunk{
		indexer.CreateTestChunk("file1.txt", "Original chunk 1", "abc123", 0),
		indexer.CreateTestChunk("file1.txt", "Original chunk 2", "abc123", 8),
		indexer.CreateTestChunk("file1.txt", "Original chunk 3", "abc123", 16),
	}

	indexedIDs, err := esIndexer.Index(ctx, testProjectID, originalChunks)
	require.NoError(t, err)
	require.Len(t, indexedIDs, 3)

	client.Client.Refresh(testIndexName).Do(ctx)
	count, err := client.Client.Count().Index(testIndexName).Do(ctx)
	require.NoError(t, err)
	require.Equal(t, int64(3), count)

	// Update the file: The file now has different chunks
	// - chunk 1: same content (should be kept)
	// - chunk 2: different content (old should be deleted, new added)
	// - chunk 3: removed entirely (should be deleted)
	// - chunk 4: new chunk (should be added)
	updatedChunks := []chunker.Chunk{
		indexer.CreateTestChunk("file1.txt", "Original chunk 1", "def456", 0),
		indexer.CreateTestChunk("file1.txt", "Updated chunk 2", "def456", 8),
		indexer.CreateTestChunk("file1.txt", "New chunk 4", "def456", 16),
	}

	indexedIDs, err = esIndexer.Index(ctx, testProjectID, updatedChunks)
	require.NoError(t, err)
	require.Len(t, indexedIDs, 3)

	// Verify final state has only the right chunks
	chunksToIndex := append([]chunker.Chunk{originalChunks[0]}, updatedChunks[1:3]...)
	verifyElasticFinalChunks(t, client, chunksToIndex)
}

func TestDeletePaths(t *testing.T) {
	esIndexer, client := setupTestIndexer(t)
	ctx := context.Background()

	chunks := []chunker.Chunk{
		indexer.CreateTestChunk("file1.txt", "Content 1", "abc123", 0),
		indexer.CreateTestChunk("file2.txt", "Content 2", "def456", 8),
		indexer.CreateTestChunk("file3.txt", "Content 3", "ghi789", 16),
	}

	indexedIDs, err := esIndexer.Index(ctx, testProjectID, chunks)
	require.NoError(t, err)
	require.Len(t, indexedIDs, 3)

	client.Client.Refresh(testIndexName).Do(ctx)
	count, err := client.Client.Count().Index(testIndexName).Do(ctx)
	require.NoError(t, err)
	require.Equal(t, int64(3), count)

	// Delete file1.txt and file3.txt
	err = esIndexer.DeletePaths(ctx, testProjectID, []string{"file1.txt", "file3.txt"})
	require.NoError(t, err)

	// Verify final state has only file2.txt chunk
	verifyElasticFinalChunks(t, client, []chunker.Chunk{chunks[1]})
}

func TestReindexing(t *testing.T) {
	esIndexer, client := setupTestIndexer(t)
	ctx := context.Background()

	// Setup the initial indexed documents

	initialChunks := []chunker.Chunk{
		indexer.CreateTestChunk("file1.txt", "First chunk", "abc123", 0),
		indexer.CreateTestChunk("file1.txt", "Second chunk", "abc123", 8),
		indexer.CreateTestChunk("file2.txt", "Another file", "def456", 16),
	}

	indexedIDs, err := esIndexer.Index(ctx, testProjectID, initialChunks)
	require.NoError(t, err)
	require.Len(t, indexedIDs, 3)
	verifyElasticFinalChunks(t, client, initialChunks)

	// Test call to ResolveReindexing if not in reindexing mode
	err = esIndexer.ResolveReindexing(ctx, testProjectID)
	require.NoError(t, err)
	verifyElasticFinalChunks(t, client, initialChunks)

	// Test reindexing, where:
	// file1.txt has been changed
	// file2.txt has been deleted

	esReindexer, err := elasticsearchIndexer.New(client, testIndexName, 2, true)
	require.NoError(t, err)

	reindexChunks := []chunker.Chunk{
		indexer.CreateTestChunk("file1.txt", "Reindexed first chunk", "abc123", 0),
		indexer.CreateTestChunk("file1.txt", "Reindexed second chunk", "abc123", 8),
	}

	reindexedIDs, err := esReindexer.Index(ctx, testProjectID, reindexChunks)
	require.NoError(t, err)

	// Test documents indexed documents before calling ResolveReindexing

	require.Len(t, reindexedIDs, 2)

	chunksBeforeResolvingReindex := append(
		reindexChunks,    // reindexed file1.txt chunks
		initialChunks[2], // file2.txt chunks
	)

	documentsBeforeResolvingReindex := queryAllDocuments(t, client)
	verifyDocumentsContent(t, chunksBeforeResolvingReindex, documentsBeforeResolvingReindex)
	for _, doc := range documentsBeforeResolvingReindex {
		if doc.Path == "file1.txt" {
			require.True(t, doc.Reindexing)
		} else {
			require.False(t, doc.Reindexing)
		}
	}

	// Test ResolveReindexing in reindexing mode

	err = esReindexer.ResolveReindexing(ctx, testProjectID)
	require.NoError(t, err)

	finalDocuments := queryAllDocuments(t, client)
	verifyDocumentsContent(t, reindexChunks, finalDocuments)
	for _, doc := range finalDocuments {
		require.False(t, doc.Reindexing)
	}
}
