package opensearch_test

import (
	"context"
	"testing"

	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/require"
	"gitlab.com/gitlab-org/gitlab-elasticsearch-indexer/internal/mode/chunk/client/opensearch"
	"gitlab.com/gitlab-org/gitlab-elasticsearch-indexer/internal/mode/chunk/types"
)

func TestNewOpenSearchClient(t *testing.T) {
	t.Parallel()

	// Test with empty URL list
	conn := types.OpenSearchConnection{
		URL: []string{},
	}
	osClient := opensearch.New(conn)
	require.NotNil(t, osClient)
	require.Equal(t, conn, osClient.Conn)

	// Test with single URL
	conn = types.OpenSearchConnection{
		URL: []string{"http://localhost:9200"},
	}
	osClient = opensearch.New(conn)
	require.NotNil(t, osClient)
	require.Equal(t, conn, osClient.Conn)

	// Test with multiple URLs
	conn = types.OpenSearchConnection{
		URL: []string{"http://os1:9200", "http://os2:9200", "http://os3:9200"},
	}
	osClient = opensearch.New(conn)
	require.NotNil(t, osClient)
	require.Equal(t, conn, osClient.Conn)
}

func TestOpenSearchClient_AWS_Configuration(t *testing.T) {
	t.Parallel()

	tests := []struct {
		name          string
		connection    types.OpenSearchConnection
		expectedError string
	}{
		{
			name: "Missing region with AWS enabled",
			connection: types.OpenSearchConnection{
				URL:       []string{"https://search-test.us-east-1.es.amazonaws.com"},
				AWS:       true,
				AccessKey: "key",
				SecretKey: "secret",
			},
			expectedError: "aws_region is required when aws is enabled",
		},
		{
			name: "Valid AWS configuration",
			connection: types.OpenSearchConnection{
				URL:       []string{"https://search-test.us-east-1.es.amazonaws.com"},
				AWS:       true,
				Region:    "us-east-1",
				AccessKey: "key",
				SecretKey: "secret",
			},
		},
	}

	for _, tt := range tests {
		t.Run(tt.name, func(t *testing.T) {
			client := opensearch.New(tt.connection)
			ctx := context.Background()

			err := client.Connect(ctx)
			if tt.expectedError != "" {
				require.Error(t, err)
				assert.Contains(t, err.Error(), tt.expectedError)
			} else {
				if err != nil {
					assert.NotContains(t, err.Error(), "aws_region is required")
				}
			}
		})
	}
}

func TestOpenSearchClient_AWS_CredentialResolution(t *testing.T) {
	t.Parallel()

	conn := types.OpenSearchConnection{
		AWS:       true,
		Region:    "us-east-1",
		AccessKey: "static-key",
		SecretKey: "static-secret",
	}

	client := opensearch.New(conn)
	require.NotNil(t, client)
	require.Equal(t, "static-key", client.Conn.AccessKey)
	require.Equal(t, "static-secret", client.Conn.SecretKey)
	require.Equal(t, "us-east-1", client.Conn.Region)
}

func TestOpenSearchClient_AWS_RequestTimeout(t *testing.T) {
	t.Parallel()

	tests := []struct {
		name           string
		requestTimeout int
	}{
		{"No timeout", 0},
		{"30 second timeout", 30},
		{"5 minute timeout", 300},
	}

	for _, tt := range tests {
		t.Run(tt.name, func(t *testing.T) {
			conn := types.OpenSearchConnection{
				URL:            []string{"https://search-test.us-east-1.es.amazonaws.com"},
				AWS:            true,
				Region:         "us-east-1",
				AccessKey:      "key",
				SecretKey:      "secret",
				RequestTimeout: tt.requestTimeout,
			}

			client := opensearch.New(conn)
			require.NotNil(t, client)
			require.Equal(t, tt.requestTimeout, client.Conn.RequestTimeout)
		})
	}
}

func TestOpenSearchClient_RoleARN_Configuration(t *testing.T) {
	t.Parallel()

	conn := types.OpenSearchConnection{
		URL:     []string{"https://test.us-east-1.es.amazonaws.com"},
		AWS:     true,
		Region:  "us-east-1",
		RoleARN: "arn:aws:iam::123456789012:role/TestRole",
	}

	client := opensearch.New(conn)
	require.NotNil(t, client)
	assert.Equal(t, "arn:aws:iam::123456789012:role/TestRole", client.Conn.RoleARN)
	assert.Empty(t, client.Conn.AccessKey)
	assert.Empty(t, client.Conn.SecretKey)
}

func TestOpenSearchClient_RoleARN_WithStaticCredentials(t *testing.T) {
	t.Parallel()

	conn := types.OpenSearchConnection{
		URL:       []string{"https://test.us-east-1.es.amazonaws.com"},
		AWS:       true,
		Region:    "us-east-1",
		AccessKey: "AKIA123",
		SecretKey: "secret123",
		RoleARN:   "arn:aws:iam::123456789012:role/TestRole",
	}

	client := opensearch.New(conn)
	require.NotNil(t, client)
	assert.Equal(t, "arn:aws:iam::123456789012:role/TestRole", client.Conn.RoleARN)
	assert.Equal(t, "AKIA123", client.Conn.AccessKey)
	assert.Equal(t, "secret123", client.Conn.SecretKey)
}
